<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Mtc\MercuryDataModels\Contracts\MinMaxFilter;

class EngineSizeMaxFilter extends EngineSizeMinFilter implements MinMaxFilter
{
    public static Collection $data;

    private ?int $min_value = null;

    public function applyFilter($query, array $selection = [])
    {
        $query->where('engine_size_cc', '<=', $selection);
    }
    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->filter(fn($range) => $this->isInBounds($range))
            ->map(fn ($range) => [
                'id' => $range,
                'value' => $range,
                'name' => $this->getSelectionName($range)
            ])
            ->values();
    }

    protected function getData(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        if (empty(self::$data)) {
            $min = floor($this->getQuery()
                ->when(
                    Settings::get('filter-apply-selections-to-results'),
                    fn($query) => $query->where($product_filtering)
                )
                ->min('engine_size_cc'));

            $ranges = collect(Config::get('automotive.filter-ranges.engine-size'));
            self::$data = $ranges
                // Up to max range including one step over current max
                ->filter(fn ($range, $i) => $range > $min
                    || ($i < $ranges->count() - 1 && $range < $min && $ranges[$i + 1] > $min));
        }

        return self::$data;
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Engine Size Up To ' . round($selection / 1000, 1) . 'L';
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/engine-up-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/engine-up-to-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'engine-up-to-' . $selection;
    }

    protected function isRangeMaximum(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'engine_max_size';
    }

    public function getOtherBound(): string
    {
        return 'engine_size_min';
    }

    public function setOtherBoundary(?array $value = null): void
    {
        if (!empty($value)) {
            $this->min_value = $value[0];
        }
    }

    private function isInBounds($value): bool
    {
        if (empty($this->min_value)) {
            return true;
        }
        return $value >= $this->min_value;
    }
}
