<?php

namespace Mtc\MercuryDataModels\Factories;

use App\Enums\MappingOperator;
use App\ExportMaps\VehicleExportMap;
use App\IntegrationRepository;
use App\Models\ExportMapTemplate;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Facades\App;

/**
 * @template TModel of \App\Models\ExportMapTemplate
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<TModel>
 */
class ExportMapTemplateFactory extends Factory
{
    /**
     * @var class-string<TModel>
     */
    protected $model = ExportMapTemplate::class;

    /** @var array<string,string> key => label from ExportMap */
    protected array $fieldMap = [];

    public function __construct(...$args)
    {
        parent::__construct(...$args);

        /** @var VehicleExportMap $provider */
        $provider = App::make(VehicleExportMap::class);
        $this->fieldMap = $provider->columns();
    }

    public function definition(): array
    {
        [$columns, $headers] = $this->pickColumnsAndHeaders($this->faker->numberBetween(2, 5));

        /** @var IntegrationRepository $repo */
        $repo = App::make(IntegrationRepository::class);
        $allSales = $repo->getAllForType('sales-channels');
        $channelKeys = $allSales->keys()
            ->values()
            ->all();

        return [
            'title' => $this->faker->unique()->sentence(3),
            'sales_channel' => $this->faker->randomElement($channelKeys),
            'is_published' => $this->faker->boolean(),
            'columns' => $columns,
            'headers' => $headers,
            'mapping_rules' => [],
        ];
    }

    protected function pickColumnsAndHeaders(int $count): array
    {
        $count = max(1, $count);
        $keys = array_keys($this->fieldMap);
        shuffle($keys);
        $columns = array_slice($keys, 0, min($count, count($keys)));
        $headers = array_map(fn ($k) => $this->fieldMap[$k] ?? $k, $columns);

        return [$columns, $headers];
    }

    public function published(): self
    {
        return $this->state(fn () => ['is_published' => true]);
    }

    public function draft(): self
    {
        return $this->state(fn () => ['is_published' => false]);
    }

    public function withColumns(int $count = 3): self
    {
        return $this->state(function () use ($count) {
            [$columns, $headers] = $this->pickColumnsAndHeaders($count);
            return compact('columns', 'headers');
        });
    }

    public function forSalesChannel(string $key): self
    {
        return $this->state(fn () => ['sales_channel' => $key]);
    }

    public function withRules(int $rules = 1, int $maxThenPerRule = 2): self
    {
        return $this->state(function (array $attributes) use ($rules, $maxThenPerRule) {
            $columns = $attributes['columns'] ?? array_keys($this->fieldMap);
            if (empty($columns)) {
                [$columns] = $this->pickColumnsAndHeaders(3);
            }

            $ops = MappingOperator::values();

            $generated = [];
            for ($i = 0; $i < max(1, $rules); $i++) {
                $whenField = $columns[array_rand($columns)];
                $op = $ops[array_rand($ops)];

                if (in_array($op, ['in','nin'], true)) {
                    $whenValue = [
                        $this->faker->randomElement(['CAR','TRUCK','VAN','SUV']),
                        $this->faker->randomElement(['NEW','USED','DEMO']),
                    ];
                } elseif (in_array($op, ['gt','gte','lt','lte'], true)) {
                    $whenValue = $this->faker->numberBetween(1, 99999);
                } else {
                    $whenValue = $this->faker->randomElement(['CAR','TRUCK','VAN','NEW','USED']);
                }

                $thenCount = $this->faker->numberBetween(1, max(1, $maxThenPerRule));
                $then = [];
                for ($j = 0; $j < $thenCount; $j++) {
                    $target = $columns[array_rand($columns)];
                    $then[] = [
                        'action' => 'set',
                        'target' => $target,
                        'value'  => $this->faker->boolean(25) ? null : $this->faker->randomElement([
                            $this->faker->word(),
                            $this->faker->numberBetween(1, 9999),
                            strtoupper($this->faker->randomLetter()),
                        ]),
                    ];
                }

                $generated[] = [
                    'when' => [
                        'field' => $whenField,
                        'op' => $op,
                        'value' => $whenValue
                    ],
                    'then' => $then,
                ];
            }

            return [
                'mapping_rules' => $generated
            ];
        });
    }
}
