<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;

class Enquiry extends EnquiryModel implements SearchableModel
{
    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->title;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return strip_tags($this->message);
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'envelope';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('crm/enquiries/view/' . $this->id, [], false);
    }

    /**
     * Related vehicle valuation
     *
     * @return BelongsTo
     */
    public function valuation(): BelongsTo
    {
        return $this->belongsTo(VehicleValuation::class);
    }

    public function getViewUrl(): string
    {
        return config('app.url') . "/crm/enquiries/view/{$this->id}";
    }

    public function getCreatedAtSinceAttribute()
    {
        return $this->created_at?->diffInDays();
    }

    public function getAttribute($key, bool $for_reports = false)
    {
        if (!$for_reports) {
            return parent::getAttribute($key);
        }


        if ($key === 'reason.name') {
            return match ($this->reason?->getMorphClass()) {
                'vehicle' => $this->reason?->vrm_condensed . ' ' . $this->reason?->title,
                'page' => $this->reason?->title,
                'valuation' => collect([
                    $this->reason?->registration,
                    $this->reason?->make,
                    $this->reason?->model,
                    $this->reason?->retail_price,
                ])->implode('|'),
                default => $this->reason?->name
            };
        } elseif (Str::startsWith($key, 'value')) {
            $field = str_replace('value.', '', $key);
            $matches = array_values(array_filter($this->details ?? [], fn($entry) => $entry['question'] == $field));
            $result = $matches[0]['answer'] ?? null;
            if (is_array($result)) {
                $result = implode(',', $result);
            }
            return $result;
        }
    }

    public function scopeSetFilters(Builder $query, array|string $filters)
    {
        if (is_string($filters)) {
            $filters = [ $filters ];
        }
        collect($filters)
            ->filter(fn ($filter) => method_exists($this, 'scope' . ucfirst($filter)))
            ->map(fn ($filter) => 'scope' . ucfirst($filter))
            ->each(fn ($scopeMethod) => $this->{$scopeMethod}($query));

        collect($filters)
            ->filter()
            ->each(function ($filter) use ($query) {
                if (str_starts_with($filter, 'enquiry-type-')) {
                    $query->whereHas(
                        'type',
                        fn($query) => $query->where('id', str_replace('enquiry-type-', '', $filter))
                    );
                } elseif (method_exists($this, 'scope' . ucfirst(Str::camel($filter)))) {
                    $scopeMethod = 'scope' . ucfirst(Str::camel($filter));
                    $this->{$scopeMethod}($query);
                }
            });
    }
}
