<?php

namespace Mtc\Klarna\Repositories;

use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Invoice;
use Mtc\Checkout\InvoiceRepository;
use Mtc\Klarna\Contracts\KlarnaRepositoryContract;
use Mtc\Klarna\Services\KlarnaApi;
use Mtc\Klarna\Traits\KlarnaInvoice;

/**
 * Class KlarnaRepository
 * @package Mtc\Klarna\Repositories
 */
class KlarnaRepository implements KlarnaRepositoryContract
{
    use KlarnaInvoice;

    /**
     * Klarna order Statuses
     */
    protected const INCOMPLETE = 'incomplete'; //order not finished
    protected const COMPLETE = 'complete'; //order has been placed

    /**
     * @var array
     */
    protected $response;

    /**
     * @var array
     */
    protected $request;

    /**
     * @var InvoiceRepository
     */
    protected $invoice_repository;

    /**
     * @var Invoice
     */
    protected $invoice;

    /**
     * @var array|void
     */
    protected $invoice_data = [];

    /**
     * @var KlarnaApi
     */
    protected $api;

    /**
     * KlarnaRepository constructor.
     * @param InvoiceRepositoryContract $invoice_repository
     */
    public function __construct(InvoiceRepositoryContract $invoice_repository)
    {
        $this->api = new KlarnaApi();
        $this->invoice_repository = $invoice_repository;
        $this->invoice = $this->invoice_repository->getModel();
        $this->invoice_data = $this->getInvoiceData();
    }

    /**
     * Get Klarna information for payment form
     *
     * @return array
     */
    public function getInformation(): array
    {
        if (!$this->invoice_repository->getId() || $this->invoice_repository->getOutstandingAmount() <= 0.01) {
            return [];
        }

        if (!session('klarna_session_id')) {
            $klarna_session = $this->api->createNewCreditSession($this->invoice_data);

            session()->put('klarna_session_id', $klarna_session['session_id'] ?? null);
            if (session('klarna_session_id') === null) {
                return [];
            }
        } else {
            $this->api->updateCreditSession(session('klarna_session_id'), $this->invoice_data);
            $klarna_session = $this->api->readCreditSession(session('klarna_session_id'));
        }

        return [
            'instance_id' => Str::random(16),
            'client_token' => $klarna_session['client_token'] ?? null,
            'payment_method_categories' => collect($klarna_session['payment_method_categories'])
                ->pluck('identifier')
        ];
    }

    /**
     * Place order
     *
     * @param $authorization_token
     * @return bool|mixed|null
     */
    public function placeOrder($authorization_token)
    {
        $response = $this->api->createOrder($authorization_token, $this->invoice_data);

        if (!isset($response['fraud_status']) || $response['fraud_status'] === 'REJECTED') {
            return false;
        }

        //Unset klarna session id
        session()->forget('klarna_session_id');

        return $this->api->getOrder($response['order_id']);
    }

    /**
     * Capture order
     *
     * @param $order_id
     * @return bool
     */
    public function captureOrder($order_id): bool
    {
        if (config('klarna.auto_capture') === true) {
            return false;
        }

        $this->api->createCapture($order_id, $this->getInvoiceCaptureData());
        return true;
    }
}
