<?php

namespace Mtc\Installer\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Mtc\Installer\Facades\Installer;

/**
 * Class CreateEnv
 *
 * @package Mtc\Installer\Console\Commands
 */
class CreateEnv extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'install:create-env';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create the environment information for site';

    /**
     * List of project types to help with installer setup
     *
     * @var array
     */
    protected static $project_types = [
        'project' => 'Project (will remove .git folders)',
        'sandbox' => 'Development environment (will leave .git folders)',
    ];

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $existing_config = $this->hasReusableConfig() ? $this->askToReuseConfig() : [];
        $project_type = $existing_config['project_type'] ?? $this->choice('Set up what type of project?', self::$project_types, 'project');

        $sandbox_on_dev_server = $project_type === 'sandbox'
            && isset($existing_config['site_domain'])
            && $this->isMtcDevServerInstance($existing_config['site_domain']);
        // Set default values for sandbox forks based on directory
        if ($sandbox_on_dev_server) {
            $existing_config = $this->setDevServerDefaults($existing_config);
        }

        $env_values = [
            'project_type' => $project_type,
            'site_domain' => $this->ask('App Url?', $existing_config['site_domain'] ?? ''),
            'site_name' => $this->ask('Name of the application?', $existing_config['site_name'] ?? ''),
            'db_host' => $existing_config['db_host'] ?? $this->ask('Database Host?', 'localhost'),
            'db_name' => $this->ask('Database Name?', $existing_config['db_name'] ?? ''),
            'db_username' => $existing_config['db_username'] ?? $this->ask('Database Username?'),
            'db_password' => $existing_config['db_password'] ?? $this->ask('Database Password?'),
        ];
        $env_values['site_id'] = Str::slug($env_values['site_name']);

        $site_config = $this->siteConfig($project_type, $existing_config);

        $this->setEnvFileValues($env_values);

        if ($project_type === 'sandbox' && $this->confirm('Do you want to store details for re-use across other installs on this server?')) {
            $this->storeReusableConfig(array_merge($env_values, $site_config));
        }

        $this->info('Setup complete. Please run `composer setup`');

    }

    /**
     * This config will be needed for the next scripts running so we store them in cache
     */
    protected function siteConfig($project_type, $existing_config)
    {
        Cache::forget('install-config');
        return Cache::remember('install-config', now()->addHour(), function () use ($project_type, $existing_config) {
            $bb_username = $existing_config['bb_username'] ?? $this->ask('Bitbucket Username?');
            $bb_password = $existing_config['bb_password'] ?? $this->secret('Bitbucket Password?');
            $dev_email = $existing_config['dev_email'] ?? $this->ask('Developer email?');
            $theme = $this->chooseTheme($existing_config['theme'] ?? null);
            $root_modules = $this->chooseRootModules($existing_config['root_modules'] ?? []);
            if (array_key_exists('cms', $root_modules)) {
                $default_cms_setup = $existing_config['default_cms_setup'] ?? $this->confirm('Do you want to use default CMS page setup?', true);
            }
            $drop_git = $project_type === 'project';
            return [
                'bb_username' => $bb_username,
                'bb_password' => $bb_password,
                'dev_email' => $dev_email,
                'theme' => $theme,
                'root_modules' => $root_modules,
                'default_cms_setup' => $default_cms_setup ?? true,
                'drop_git' => $drop_git,
            ];
        });

    }

    /**
     * Choose which theme to install
     *
     * @return bool
     */
    protected function chooseTheme($default = null)
    {
        $themes = Installer::availableComponents('themes');
        $theme_names = $themes->pluck('name')->push('other');

        if (!empty($default)) {
            $suggestion = $themes->where('full_name', $default)->first();
            $suggestion = $suggestion['name'] ?? $theme_names->first();
        } else {
            $suggestion = $theme_names->first();
        }

        $theme = $this->choice('Which Theme to install?', $theme_names->toArray(), $suggestion);
        if ($theme !== 'other') {
            $selected_theme = $themes->where('name', $theme)->first();
            return $selected_theme['full_name'];
        }

        $theme = $this->ask('Please specify repo name for the theme (e.g. mtcmedia/theme)?');
        $theme_url = Installer::repoUrl($theme);
        if ($this->confirm("Do you want to clone the theme from {$theme_url}?", true) === false) {
            return false;
        }
        return $theme_url;
    }


    /**
     * Choose which root modules to install
     *
     * @return Collection
     */
    protected function chooseRootModules($defaults = [])
    {
        return collect(config('installer.root_modules'))
            ->reject(function ($module) {
                // Core should already be installed
                return $module === 'core';
            })
            ->filter(function ($module, $module_path) use ($defaults) {
                return $this->confirm("Do you want to install {$module_path}?", in_array($module, $defaults));
            })->toArray();
    }

    /**
     * Write info to .env file
     *
     * @param $data
     */
    protected function setEnvFileValues($data)
    {
        if (!File::exists(base_path('.env'))) {
            File::copy(base_path('.env.example'), base_path('.env'));
        }

        $env_file = File::get(base_path('.env'));
        $env_file = preg_replace([
            '/APP_NAME=(.*)/',
            '/APP_URL=(.*)/',
            '/SITE_ID=(.*)/',
            '/DB_HOST=(.*)/',
            '/DB_DATABASE=(.*)/',
            '/DB_USERNAME=(.*)/',
            '/DB_PASSWORD=(.*)/',
        ], [
            'APP_NAME="' . $data['site_name'] . '"',
            'APP_URL=' . $data['site_domain'],
            'SITE_ID=' . $data['site_id'],
            'DB_HOST=' . $data['db_host'],
            'DB_DATABASE=' . $data['db_name'],
            'DB_USERNAME=' . $data['db_username'],
            'DB_PASSWORD="' . $data['db_password'] . '"',
        ], $env_file);

        File::put(base_path('.env'), $env_file);
    }

    /**
     * Check if reusable config exists
     */
    protected function hasReusableConfig()
    {
        return File::exists(dirname(base_path()) . '/mtc_installer_defaults.json');
    }


    /**
     * Get reusable config values
     */
    protected function getReusableConfig()
    {
        return json_decode(File::get(dirname(base_path()) . '/mtc_installer_defaults.json'), true);
    }

    /**
     * Store the reusable config
     *
     * @param $data
     */
    protected function storeReusableConfig($data)
    {
        File::put(dirname(base_path()) . '/mtc_installer_defaults.json', json_encode($data));
    }

    /**
     * Check if user wants to reuse config values
     */
    protected function askToReuseConfig()
    {
        $config_values = $this->getReusableConfig();
        $this->info('We found config on server with these values:');
        collect($config_values)
            ->filter(function ($value, $key) {
                return in_array($key, config('installer.skippable_values', []));
            })
            ->each(function ($value, $key) {
                if ($key === 'bb_password') {
                    $value = str_repeat('*', strlen($value));
                }
                $this->info("  {$key} => {$value}");
            });
        if ($this->confirm('Do you want to re-use these values?', true)) {
            return $config_values;
        }
        return [];
    }

    /**
     * Get the directory of the install
     *
     * @return string|string[]
     */
    protected function getProjectDirectory()
    {
        // This seems a bit dirty we look for filename but expect directory name
        // This is working because pathinfo method doesn't actually check file types
        // and simply assumes last segment of the path is the file
        return pathinfo(base_path(), PATHINFO_FILENAME);
    }

    /**
     * Check if this is a mtc-dev-server instance
     * This allows setting some expectations for defaults
     *
     * @param $domain
     * @return bool
     */
    protected function isMtcDevServerInstance($domain = '')
    {
        return strpos($domain, 'mtcdevserver') !== false;
    }

    /**
     * Set defaults for our naming conventions in cpanel sturcture
     *
     * @param $existing_config
     * @return mixed
     */
    protected function setDevServerDefaults($existing_config)
    {
        // Skip changing things in public_html
        if ($this->getProjectDirectory() === 'public_html') {
            return $existing_config;
        }

        $trimmed_path = str_replace('public_', '', $this->getProjectDirectory());

        // Site name = directory name
        $existing_config['site_name'] = $trimmed_path;

        // DB = account_fork
        $account_name = pathinfo(pathinfo(base_path(), PATHINFO_DIRNAME), PATHINFO_FILENAME);
        $existing_config['db_name'] = $account_name . '_' . $trimmed_path;

        // Domain = replace subdomain with current directory
        if (!empty($existing_config['site_domain'])) {
            $url_segments = explode('.', parse_url($existing_config['site_domain'], PHP_URL_HOST));
            $url_segments[0] = $trimmed_path;
            $existing_config['site_domain'] = "http://" . implode('.', $url_segments);
        }

        return $existing_config;
    }
}
