<?php

namespace Mtc\Installer;

use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Mtc\Foundation\Contracts\InstallableComponent;
use Symfony\Component\Process\Process;
use Illuminate\Foundation\PackageManifest;

/**
 * Class ComponentInstaller
 *
 * @package Mtc\Installer
 */
class ComponentInstaller
{
    /**
     * @var PackageManifest
     */
    protected $package_handler;

    /**
     * List of assets to publish
     *
     * @var array
     */
    protected $publish = [];

    /**
     * List of seeds to run
     *
     * @var array
     */
    protected $seeds = [];

    /**
     * List of npm packages to install
     *
     * @var array
     */
    protected $npm_dependencies = [];

    /**
     * Whether npm compilation should be run
     *
     * @var bool
     */
    protected $run_npm = false;

    /**
     * ComponentInstaller constructor.
     *
     * @param PackageManifest $package_manifest
     */
    public function __construct()
    {
        $this->package_handler = App::make(PackageManifest::class);
    }

    /**
     * Run the installation process
     *
     * @param Command $command
     * @return void
     * @throws \Exception
     */
    public function run(Command $command): void
    {
        $command->info('Starting Installer');
        $installers = $this->prepare();
        $command->info('Running Migrations');
        $command->callSilent('migrate');

        $command->info('Publishing Data');
        collect($this->publish)
            ->each (function ($data_to_publish) use ($command) {
                $command->callSilent('vendor:publish', $data_to_publish);
            });

        $command->info('Seeding Data');
        collect($this->seeds)
            ->each (function ($seeder_class_name) use ($command) {
                $command->call('db:seed', [
                    '--class' => $seeder_class_name
                ]);
            });

        $command->info('Installing npm packages');
        if (!empty($this->npm_dependencies)) {
            $npm_command = collect($this->npm_dependencies)
                ->map(function ($version, $name) {
                    return $name . '@' . $version;
                })
                ->push('--save')
                ->prepend('install')
                ->prepend('npm')
                ->toArray();

            $process = (new Process($npm_command));
            $process->setTimeout(900);
            $process->run();
            if ($process->isSuccessful() === false) {
                $command->info($process->getErrorOutput());
                throw new \Exception('Failed to install assets');
            }
        }

        $command->info('Running asset compilation');
        if ($this->run_npm) {
            $env = $this->isProductionEnvironment() ? 'production' : 'dev';
            $process = (new Process([ 'npm', 'run',  $env ]));
            $process->setTimeout(900);
            $process->run();
            if ($process->isSuccessful() === false) {
                $command->info($process->getErrorOutput());
                throw new \Exception('Failed to compile assets');
            }
        }

        $command->callSilent('cache:clear');

        $this->markInstallersComplete($installers);
        $command->info('Finishing Installer');
        collect($installers)->each(function ($installer) use ($command) {
            $command->info(" - {$installer} set up");
        });
    }

    /**
     * Prepare installers to find what needs to be installed
     *
     * @return Collection
     */
    protected function prepare(): Collection
    {
        return collect($this->findInstallers())
            ->filter()
            ->reject(function ($installer_class) {
                return $this->checkIfAlreadyInstalled($installer_class);
            })
            ->each(function ($installer_class) {
                $this->setInstaller($installer_class);
            });
    }
    /**
     * Find the registered installers
     *
     * @return Collection
     */
    protected function findInstallers():Collection
    {
        return collect($this->package_handler->config('installers'))
            ->flatten(1);
    }

    /**
     * Set up Installer to run
     *
     * @param $installer_class
     * @return bool|void
     */
    protected function setInstaller($installer_class): void
    {
        /** @var InstallableComponent $installer */
        $installer = App::make($installer_class);
        if ($this->installerNotSupported($installer)) {
            return;
        }

        if (!empty($installer->publish())) {
            $this->publish = array_merge($this->publish, $installer->publish());
        }

        if (!empty($installer->npmDependencies())) {
            $this->npm_dependencies = array_merge($this->npm_dependencies, $installer->npmDependencies());
        }

        if (!empty($installer->seed())) {
            $this->seeds[] = $installer->seed();
        }

        if ($installer->shouldRunNpm()) {
            $this->run_npm = true;
        }
    }

    /**
     * Check if installer has completed
     *
     * @param $installer_class
     * @return bool
     */
    protected function checkIfAlreadyInstalled($installer_class): bool
    {
        return File::exists($this->getFileNameForInstaller($installer_class));
    }

    /**
     * Mark installer classes as run
     *
     * @param Collection $classes_to_set_completed
     */
    protected function markInstallersComplete(Collection $classes_to_set_completed): void
    {
        if (File::exists(storage_path('framework/component-installs/')) === false) {
            File::makeDirectory(storage_path('framework/component-installs/'));
        }

        $classes_to_set_completed->each(function ($installer_class) {
            File::put($this->getFileNameForInstaller($installer_class), Carbon::now()->format('Y-m-d H:i:s'));
        });
    }

    /**
     * Get the file name that represents an installer class
     *
     * @param $installer_class
     * @return string
     */
    protected function getFileNameForInstaller($installer_class): string
    {
        return storage_path('framework/component-installs/' . str_replace('\\', '_', $installer_class) . '.installed');
    }

    /**
     * Check if the installer is supported
     *
     * @param $installer
     * @return bool
     */
    protected function installerNotSupported($installer): bool
    {
        return ($installer instanceof InstallableComponent) === false;
    }

    /**
     * Check if production mode is set
     *
     * @return bool
     */
    protected function isProductionEnvironment(): bool
    {
        return in_array(config('app.env'), ['demo', 'production']);
    }
}