<?php

namespace Mtc\Foundation;

use Illuminate\Database\Eloquent\Model;

/**
 * Class Address
 *
 * Foundation address that can implements base functionality across different address usages
 *
 * @package Mtc\Foundation
 */
class Address extends Model
{
    /**
     * Address type constants
     */
    const TYPE_SHIPPING = 'shipping';
    const TYPE_BILLING = 'billing';
    const TYPE_COLLECTION = 'collection';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id',
        'owner_country',
    ];

    /**
     * Address blueprint - empty address array
     *
     * @var array
     */
    public static $blueprint = [
        'type' => '',
        'first_name' => '',
        'last_name' => '',
        'address1' => '',
        'address2' => '',
        'city' => '',
        'county' => '',
        'state' => '',
        'postcode' => '',
        'country' => '',
    ];

    /**
     * Fields that identify address (excluding name)
     *
     * @var array
     */
    public static $address_fields = [
        'address1',
        'address2',
        'city',
        'county',
        'state',
        'postcode',
        'country',
    ];

    protected static function boot()
    {
        parent::boot();
        if (config('foundation.load_full_country_names') === true) {
            self::addGlobalScope(function ($query) {
                return $query->with('ownerCountry');
            });
        }
    }

    /**
     * Relationship to the country that has this code
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function ownerCountry()
    {
        return $this->belongsTo(Country::class, 'country', 'code');
    }

    /**
     * Get all filled address fields
     *
     * @return \Illuminate\Support\Collection
     */
    public function getFullAddressAttribute()
    {
        return collect($this->only(self::$address_fields))->filter()
            ->map(function($value, $key) {
                return $key === 'country' ? $this->formatCountryAttributeValue() : $value;
            })
            ->reject(function ($value) {
                return $value === 'NA';
            });
    }

    /**
     * Get country name value
     *
     * @return string
     */
    private function formatCountryAttributeValue()
    {
        return config('foundation.load_full_country_names') === true && !empty($this->ownerCountry->name) ?
            $this->ownerCountry->name : $this->country;
    }

    /**
     * Get Address fields separated with comma
     *
     * @return string
     */
    public function getFormattedAddressAttribute()
    {
        return $this->getFullAddressAttribute()->implode(', ');
    }

    /**
     * Quick access to customers full name
     *
     * @return string
     */
    public function getCustomerNameAttribute()
    {
        return $this->first_name . ' ' . $this->last_name;
    }
}
