<?php

namespace Mtc\Filter\Contracts;

use Closure;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\Shop\Item\Custom;
use Mtc\Shop\Item\CustomMultiple;

abstract class IsCustomFieldFilter extends IsFilter
{
    /**
     * Get the field column/db_name name
     *
     * @return string
     */
    abstract public function getFieldColumnName(): string;

    /**
     * Is this custom field stored in custom_multiple format
     *
     * @return string
     */
    abstract public function isMultiple(): bool;

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return $this->isMultiple()
            ? CustomMultiple::class
            : Custom::class;
    }

    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        if ($this->isMultiple()) {
            $query->whereHas('customMultiple', function (Builder $custom_query) use ($selection) {
                $custom_query->whereIn('value', $selection)
                    ->where('field', $this->getFieldColumnName());
            });
        } else {
            $query->whereHas('custom', function (Builder $custom_query) use ($selection) {
                $custom_query->whereIn($this->getFieldColumnName(), $selection);
            });
        }
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->isMultiple()
            ? $this->multipleValueResults($product_filtering, $limit)
            : $this->singleValueResults($product_filtering, $limit);
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->getAttribute($this->getFieldColumnName()) ?? '';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return $this->isMultiple() ? 'value' : $this->getFieldColumnName();
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return $this->getNameAttribute();
    }

    /**
     * Get available results of this filter type when values are stored in items_custom_multiple
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @return Collection
     */
    protected function multipleValueResults(Closure $product_filtering, int $limit): Collection
    {
        $column = $this->getFieldColumnName();
        return CustomMultiple::query()
            ->selectRaw(DB::raw("DISTINCT(value)"))
            ->where('field', $column)
            ->whereHas('item', function ($item_query) use ($product_filtering) {
                return $product_filtering($item_query);
            })

            ->when(config('filter.get_filter_result_count'), function ($query) use ($product_filtering, $column) {
                $query->orderByDesc('result_count')
                    ->addSelect([
                        'result_count' => CustomMultiple::query()
                            ->from('items_custom_multiple', 'items_custom_multiple2')
                            ->selectRaw('count(*)')
                            ->whereHas('item', function ($item_query) use ($product_filtering) {
                                $this->adjustRelationshipName($item_query, 'items_custom_multiple2.item_id');
                                $product_filtering($item_query);
                            })
                            ->whereRaw(DB::raw("items_custom_multiple.value = items_custom_multiple2.value"))
                            ->where('items_custom_multiple2.field', $column),

                    ]);
            })
            ->when($limit > 0, function ($query) use ($limit) {
                $query->limit($limit);
            })
            ->get();
    }

    /**
     * Get available results of this filter type when value is stored in items_custom
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    protected function singleValueResults(Closure $product_filtering, int $limit): Collection
    {
        $column = $this->getFieldColumnName();
        return Custom::query()
            ->selectRaw(DB::raw("DISTINCT($column)"))
            ->whereHas('item', function ($item_query) use ($product_filtering) {
                return $product_filtering($item_query);
            })
            ->when(config('filter.get_filter_result_count'), function ($query) use ($product_filtering, $column) {
                $query->orderByDesc('result_count')
                    ->addSelect([
                        'result_count' => Custom::query()
                            ->from('items_custom', 'items_custom2')
                            ->selectRaw('count(*)')
                            ->whereHas('item', function ($item_query) use ($product_filtering) {
                                $this->adjustRelationshipName($item_query, 'items_custom2.item_id');
                                $product_filtering($item_query);
                            })
                            ->whereRaw(DB::raw("items_custom.$column = items_custom2.$column")),

                    ])
                    ->orderByDesc('result_count');
            })
            ->when($limit > 0, function ($query) use ($limit) {
                $query->limit($limit);
            })
            ->get();
    }
}
