<?php

namespace Mtc\Filter\Contracts;

use Closure;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;

abstract class IsFilter
{
    protected bool $has_load_more = false;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    abstract public function applyFilter($query, array $selection = []);

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    abstract public function getResults(Closure $product_filtering, int $limit, array $selections = []): Collection;

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    abstract public function getModel(): string;

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    abstract public function title(): string;

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    abstract public function modelSlug(Model $model): string;

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection->map(fn(Model $model) => [
                'id' => $model[$this->getIdAttribute()],
                'name' => $model[$this->getNameAttribute()],
                'count' => $model->result_count,
            ]),
        ];
    }

    /**
     * Specify if this filter should show load more
     *
     * @param Collection $collection
     * @return bool
     */
    protected function hasLoadMore(Collection $collection): bool
    {
        return $this->has_load_more;
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'id';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'name';
    }

    /**
     * Specify UI component type used for this filter
     *
     * @return string
     */
    public function uiComponentType(): string
    {
        return 'checkbox-filter';
    }

    /**
     * Specify whether model should be re-indexed in the index
     * @return bool
     */
    public function reindexModel(): bool
    {
        return true;
    }

    /**
     * Whether to allow setting entry order in admin UI for this filter type
     *
     * @return bool
     */
    public function allowIndexOrdering(): bool
    {
        return false;
    }

    /**
     * Need to "hack" relationship to change the related column name
     * This is due to repeated table name usage in main query and sub-query
     *
     * @param Builder $query
     * @param string $name
     */
    protected function adjustRelationshipName(Builder $query, string $name): void
    {
        $relationship = $query->getQuery()->wheres[0];
        $relationship['first'] = $name;
        $query->getQuery()->wheres[0] = $relationship;
    }
}
