<?php

namespace Mtc\Filter\Seo;

use Mtc\Core\Seo\Defaults;
use Mtc\Filter\Contracts\FilterInstance;
use Mtc\Filter\Contracts\SeoMatchingMechanism;
use Mtc\Shop\Brand;
use Mtc\Shop\Category;

class MatchSeoDefaults implements SeoMatchingMechanism
{
    /**
     * Perform a check if this mechanism should be used for setting up SEO
     * data for the page
     *
     * @param FilterInstance $filter
     * @return bool
     */
    public function matchesRequest(FilterInstance $filter): bool
    {
        return Defaults::query()
            ->where('path', 'Filter')
            ->exists();
    }

    /**
     * Obtain SEO data for given filter object/page
     *
     * @param FilterInstance $filter
     * @return array
     */
    public function getData(FilterInstance $filter): array
    {
        $pattern = Defaults::query()
            ->where('path', 'Filter')
            ->first();

        return [
            'title' => $this->processText($pattern->title ?? '', $filter),
            'description' => $this->processText($page->description ?? '', $filter),
        ];
    }

    /**
     * Process replacement tags
     *
     * @param string $input
     * @param FilterInstance $filter
     * @return string
     */
    protected function processText(string $input, FilterInstance $filter): string
    {
        $tags = $this->replacementTags($filter);
        $replaced = str_replace(array_keys($tags), $tags, $input);

        //remove unmatched keywords and resulting whitespace surplus
        $replaced = preg_replace('/{[^}]+}?/', '', $replaced);
        $replaced = preg_replace('/(\s+)/', ' ', $replaced);
        return preg_replace('/ (- )+/', ' - ', $replaced);
    }

    /**
     * Match tags with values
     *
     * @param FilterInstance $filter
     * @return array
     */
    protected function replacementTags(FilterInstance $filter)
    {
        return [
            '{SITE_NAME}', config('app.name'),
            '{PAGE}' => $filter->getCurrentPage(),
            '{BRAND}' => $this->getBrandName($filter->getSelections('brand')),
            '{CATEGORIES}' => $this->getAllCategoryNames($filter->getSelections('category')),
            '{CATEGORY}' => $this->getMainCategoryName($filter->getSelections('category')),
        ];
    }

    /**
     * Find brand name
     *
     * @param array $selected_brands
     * @return string
     */
    protected function getBrandName(array $selected_brands): string
    {
        if (empty($selected_brands)) {
            return '';
        }

        return Brand::query()->find(array_shift($selected_brands))
            ->name ?? '';
    }

    /**
     * Match multiple categories
     *
     * @param array $selected_categories
     * @return string
     */
    protected function getAllCategoryNames(array $selected_categories): string
    {
        if (empty($selected_categories)) {
            return '';
        }

        return Category::query()
            ->whereIn('id', $selected_categories)
            ->pluck('name')
            ->implode(' ');
    }

    /**
     * Match main category
     *
     * @param array $selected_categories
     * @return string
     */
    protected function getMainCategoryName(array $selected_categories): string
    {
        if (empty($selected_categories)) {
            return '';
        }
        $main_category = array_pop($selected_categories);
        return Brand::query()->find($main_category)
                ->name ?? '';
    }
}
