<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Filter\FilterIndex;
use Mtc\Shop\Item;
use Mtc\Shop\Item\Size;

class ColourFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('sizes', function (Builder $brand_query) use ($selection) {
            $brand_query->whereIn('colour', $selection);
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        /** @var LengthAwarePaginator $results */
        $results = FilterIndex::query()
            ->where('filter_type', 'colour')
            ->whereHas('colours', function ($query) use ($product_filtering) {
                $query->active()->whereHas('item', $product_filtering);
            })
            ->when(config('filter.get_filter_result_count'), function ($query) use ($product_filtering) {
                $query->orderByDesc('result_count')
                    ->addSelect([
                        'result_count' => Item::query()
                            ->selectRaw('count(*)')
                            ->where(fn($item_query) => $product_filtering($item_query))
                            ->whereHas('sizes', function ($size_query) {
                                $size_query->active()
                                    ->where(DB::raw('colour = filter_id'));
                            }),
                    ]);
            })
            ->paginate($limit, ['*'], 'sidebar-page');

        $this->has_load_more = $results->lastPage() > 1 && $limit > 0;
        return collect($results->all());
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Size::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.colour');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->colour;
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return $for_index ? 'colour' : 'filter_id';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return $for_index ? 'colour' : 'name';
    }

    /**
     * Whether to allow setting entry order in admin UI for this filter type
     *
     * @return bool
     */
    public function allowIndexOrdering(): bool
    {
        return true;
    }
}
