<?php

namespace Mtc\Filter\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\Filter\Contracts\IsFilter;
use Mtc\Shop\Category;
use Mtc\Shop\Item;

class CategoryFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder|QueryBuilder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('categories', function (Builder $query) use ($selection) {
            $query->whereIn('categories.id', $selection)
                ->active();
        });
    }

    /**
     * Get available results of this filter type
     *
     * @param Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return Category::query()
            ->active()
            ->whereHas('items', function ($item_query) use ($product_filtering) {
                return $product_filtering($item_query);
            })
            ->where('sub_id', $this->getParentCategory($selections))
            ->when(config('filter.get_filter_result_count'), function ($query) use ($product_filtering) {
                $query->orderByDesc('result_count')
                    ->addSelect([
                    'result_count' => Item::query()
                        ->selectRaw('count(*)')
                        ->join('items_categories', 'item_id', '=', 'items.id')
                        ->where(function ($item_query) use ($product_filtering) {
                            $product_filtering($item_query);
                        })
                        ->where(DB::raw('categories.id = `cat_id`')),
                ]);
            })
            ->get();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Category::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return __('filter::filter.labels.category');
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->name;
    }

    /**
     * Specify UI component type used for this filter
     *
     * @return string
     */
    public function uiComponentType(): string
    {
        return 'tree-filter';
    }

    /**
     * Determine the display level by setting which parent should be used for results
     *
     * @param array $selections
     * @return int
     */
    protected function getParentCategory(array $selections)
    {
        // No selections - top level
        if (empty($selections)) {
            return 0;
        }

        $last_category = array_pop($selections);
        $has_children = Category::query()
            ->where('sub_id', $last_category)
            ->exists();

        // If current level has children then parent = last selection
        // Alternatively show parent of the last selection to show siblings of last selection
        return $has_children
            ? $last_category
            : (Category::query()->find($last_category)->sub_id ?? 0);
    }

    public function format(Collection $collection): array
    {
        $result = parent::format($collection);
        $result['parent'] = $this->getCollectionParent($collection);
        return $result;
    }

    protected function getCollectionParent(Collection $collection)
    {
        /** @var Category|null $first */
        $first = $collection->first();
        return $first && $first->parent ?
            [
                'id' => $first->parent[$this->getIdAttribute()],
                'name' => $first->parent[$this->getNameAttribute()],
            ]
            : null;
    }
}
