<?php

namespace Mtc\Events;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Storage;
use Mtc\Events\Factories\TicketFactory;
use Mtc\Members\Member;
use Mtc\Money\HasPrices;

/**
 * Class Ticket
 * @package Mtc\Tickets
 */
class Ticket extends Model
{
    use HasPrices;
    use SoftDeletes;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'event_tickets';

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'event_id',
        'purchase_type',
        'purchase_id',
        'reference',
        'customer_name',
        'member_id',
        'email',
        'contact_number',
    ];

    /**
     * Cast variables to types
     *
     * @var string[]
     */
    protected $casts = [
        'redeemed_at' => 'datetime'
    ];

    /**
     * Event relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function ticketEvent()
    {
        return $this->belongsTo(Event::class, 'event_id');
    }

    /**
     * Purchase from which the ticket was generated
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function purchase()
    {
        return $this->morphTo();
    }

    /**
     * Member relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class);
    }

    /**
     * Get the ticket file location
     *
     * @return string
     */
    public function getTicketFileLocation()
    {
        return 'tickets/' . $this->getTicketFilename();
    }

    /**
     * Get the ticket filename
     *
     * @return string
     */
    public function getTicketFilename($extension = '.pdf')
    {
        return $this->reference . $extension;
    }

    /**
     * @return mixed
     */
    public function getQRCodePath()
    {
        return 'ticket-qr-codes/' . $this->getTicketFilename('.png');
    }

    /**
     * @return mixed
     */
    public function getQRCodeData($encoded = true)
    {
        if ($encoded) {
            return base64_encode(Storage::get($this->getQRCodePath()));
        }
        return Storage::get($this->getQRCodePath());
    }
}
