<?php

namespace Mtc\Events\Listeners;

use Illuminate\Support\Facades\Mail;
use Mtc\Events\Event;
use Mtc\Events\Mail\TicketEmail;
use Mtc\Events\TicketRepository;
use Mtc\Orders\Events\OrderPaid;

class CreateTicketsFromPaidOrder
{
    /**
     * @var TicketRepository
     */
    protected $repository;

    /**
     * CreateTicketsFromPaidOrder constructor.
     *
     * @param TicketRepository $repository
     */
    public function __construct(TicketRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Handle the event.
     *
     * @param OrderPaid $event
     * @return void
     */
    public function handle($event)
    {
        $tickets = collect([]);
        collect($event->getOrder()->items)
            ->filter(function ($order_item) {
                return $this->isTicketItem($order_item);
            })
            ->map(function ($order_item) use ($event, $tickets) {
                foreach (range(1, $order_item->quantity) as $ticket_in_order_item) {
                    $ticket = $this->repository->create($order_item['purchasable_id'], $event->getOrder(), 'order');
                    $tickets->push($ticket);
                }
            });

        if ($tickets->isNotEmpty()) {
            $this->sendCustomerTicketEmail($event->getOrder()->email, $tickets);
        }
    }

    /**
     * Check if order item is
     * @param $order_item
     * @return bool
     */
    protected function isTicketItem($order_item): bool
    {
        return in_array($order_item->purchasable_type, ['event', Event::class]);
    }

    /**
     * Send email with tickets attached
     *
     * @param $recipient
     * @param $tickets
     */
    protected function sendCustomerTicketEmail($recipient, $tickets)
    {
        Mail::to($recipient)->send(new TicketEmail($tickets));
    }
}
