<?php

namespace Mtc\Enquiries;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Mtc\Enquiries\Contracts\EnquiryRepositoryContract;

class EnquiryRepository implements EnquiryRepositoryContract
{
    public function getList(Request $request): LengthAwarePaginator
    {
        return Enquiry::query()
            ->when($request->has('enquiry_type'), function ($query) use ($request) {
                $query->where('enquiry_type', $request->input('enquiry_type'));
            })
            ->when($request->has('status'), function ($query) use ($request) {
                $query->where('status_id', $request->input('status'));
            })
            ->when($request->has('manager'), function ($query) use ($request) {
                $query->where('user_id', $request->input('manager'));
            })
            ->latest()
            ->paginate(25);
    }

    /**
     * @param int $id
     * @return Enquiry|Model
     */
    public function find(int $id): Enquiry
    {
        return Enquiry::query()->findOrFail($id);
    }

    public function create(string $name, string $email, array $data)
    {
        try {
            return Enquiry::create([
                'name' => $name,
                'email' => $email,
                'data' => $data,
                'status_id' => Config::get('enquiries.default_status'),
            ]);
        } catch (\Exception $exception) {
            Log::error('Failed to save Enquiry', [$exception]);
            return false;
        }
    }

    public function assign(int $id, int $user_id): bool
    {
        return Enquiry::query()
            ->where('id', $id)
            ->update([
                'user_id' => $user_id
            ]);
    }

    public function setStatus(int $id, int $status_id): bool
    {
        return Enquiry::query()
            ->where('id', $id)
            ->update([
                'status_id' => $status_id
            ]);
    }

    public function addNote(int $id, string $note, int $user_id)
    {
        $enquiry = Enquiry::query()->findOrFail($id);
        $enquiry->notes()
            ->create([
                'note' => $note,
                'user_id' => $user_id,
            ]);
    }

    public function getStatuses(): array
    {
        return collect(Config::get('enquiries.statuses', []))
            ->map(function ($value) {
                return __($value);
            })->toArray();
    }
}
