<?php

namespace Mtcmedia\EncryptionModule\Services;

use Illuminate\Contracts\Filesystem\Factory as FilesystemFactory;
use Illuminate\Http\File;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\StreamedResponse;

class FileEncrypter
{
    public function __construct(
        private readonly FilesystemFactory $filesystem,
        private readonly string $suffix = '.enc',
        private readonly bool $enabled = true
    ) {
    }

    /**
     * Encrypt contents and store them on disk.
     *
     * @param  UploadedFile|File  $file
     * @param  string  $disk
     * @param  string  $directory
     * @param  string|null  $filename
     * @return array<string, mixed>
     */
    public function storeEncryptedFile(UploadedFile|File $file, string $disk, string $directory, ?string $filename = null): array
    {
        $adapter = $this->filesystem->disk($disk);

        $directory = trim($directory, '/');
        $originalExtension = strtolower($file->getClientOriginalExtension());
        $safeName = $filename ?: $this->buildSafeName($file);
        $storedName = $safeName . '.' . $originalExtension;

        $contents = file_get_contents($file->getRealPath());
        if ($contents === false) {
            throw new \RuntimeException('Unable to read uploaded file contents.');
        }

        $payload = $this->enabled ? Crypt::encrypt($contents) : $contents;
        $finalName = $this->enabled ? $storedName . $this->suffix : $storedName;
        $path = ltrim(($directory !== '' ? $directory . '/' : '') . $finalName, '/');

        $adapter->put($path, $payload);

        return [
            'path' => $path,
            'disk' => $disk,
            'stored_name' => $finalName,
            'original_name' => $file->getClientOriginalName(),
            'size' => $file->getSize(),
            'encrypted' => $this->enabled,
            'extension' => $originalExtension,
        ];
    }

    /**
     * Retrieve decrypted file contents as a string.
     */
    public function getDecryptedContents(string $disk, string $path): string
    {
        $contents = $this->filesystem->disk($disk)->get($path);

        return $this->enabled ? Crypt::decrypt($contents) : $contents;
    }

    /**
     * Stream decrypted contents via HTTP response.
     *
     * @param  string  $disk
     * @param  string  $path
     * @param  string  $downloadName
     * @param  array<string, string>  $headers
     */
    public function streamDownload(string $disk, string $path, string $downloadName, array $headers = []): StreamedResponse
    {
        $headers = array_merge([
            'Content-Type' => 'application/octet-stream',
            'Content-Disposition' => 'attachment; filename="' . addslashes($downloadName) . '"',
        ], $headers);

        return response()->stream(function () use ($disk, $path) {
            $contents = $this->getDecryptedContents($disk, $path);
            echo $contents;
        }, 200, $headers);
    }

    /**
     * Remove encryption suffix from stored filenames.
     */
    public function stripSuffix(string $filename): string
    {
        if ($this->enabled && str_ends_with($filename, $this->suffix)) {
            return substr($filename, 0, -strlen($this->suffix));
        }

        return $filename;
    }

    private function buildSafeName(UploadedFile|File $file): string
    {
        $name = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $slug = Str::slug($name, '_');

        return $slug !== '' ? $slug : Str::random(12);
    }
}
