<?php

namespace Mtc\Crm\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\Crm\Models\EnquiryStatus;
use Mtc\Crm\Models\EnquiryType;
use Mtc\Crm\Models\Form;
use Mtc\Crm\Tests\AsUser;
use Mtc\Crm\Tests\TestCase;

class FormControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testListForms()
    {
        $this->asUser();
        $response = $this->get(route('forms.index'));

        $this->assertEquals(200, $response->status());
        $body = $response->getData(true);
        $this->assertArrayHasKey('current_page', $body);
        $this->assertArrayHasKey('data', $body);
        $this->assertEquals(0, count($body['data']));

        Form::factory()->create(['name' => 'foo']);
        Form::factory()->create(['name' => 'baz']);
        Form::factory()->create(['name' => 'bar']);

        $response = $this->getJson(route('forms.index'));

        $body = $response->getData(true);
        $this->assertEquals(3, count($body['data']));
    }

    public function testShowForm()
    {
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $response = $this->getJson(route('forms.show', $form->id));

        $this->assertEquals(200, $response->status());
        $body = $response->getData();
        $this->assertEquals('foo', $body->name);

        $this->assertArrayHasKey('available_actions', $response->json());
        $this->assertArrayHasKey('send-copy-via-mail', $response->json('available_actions'));
        $this->assertArrayHasKey('name', $response->json('available_actions.send-copy-via-mail'));
        $this->assertArrayHasKey('fields', $response->json('available_actions.send-copy-via-mail'));
        $this->assertArrayHasKey('enabled', $response->json('available_actions.send-copy-via-mail.fields'));
        $this->assertEquals(1, $body->is_active);
    }

    public function testUpdateForm()
    {
        $this->asUser();
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $status = EnquiryStatus::factory()->create();
        $type = EnquiryType::factory()->create();

        $response = $this->putJson(route('forms.update', $form->id), [
            'success_message' => 'Foo Baz Bar',
            'initial_status_id' => $status->id,
            'type_id' => $type->id,
            'initial_assignee' => 10,
        ]);

        $response->assertStatus(200);

        $form->refresh();

        $this->assertEquals('Foo Baz Bar', $form->success_message);
        $this->assertEquals($status->id, $form->initial_status_id);
        $this->assertEquals($type->id, $form->type_id);
        $this->assertEquals(10, $form->initial_assignee);
    }

    public function testUpdateFormWithSendCopyActionGood()
    {
        $this->asUser();
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $status = EnquiryStatus::factory()->create();
        $type = EnquiryType::factory()->create();

        $response = $this->putJson(route('forms.update', $form->id), [
            'success_message' => 'Foo Baz Bar',
            'initial_status_id' => $status->id,
            'type_id' => $type->id,
            'initial_assignee' => 10,
            'actions' => [
                'send-copy-via-mail' => [
                    'enabled' => true,
                    'recipient' => 'john@example.com',
                    'enquiry_subject' => 'Lorem ipsum',
                    'enquiry_content' => 'foo baz bar'
                ]
            ],
        ]);

        $response->assertStatus(200);
        $this->assertTrue($form->actions()->where('action_name', 'send-copy-via-mail')->exists());
    }

    public function testUpdateFormWithSendCopyActionBad()
    {
        $this->asUser();
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $status = EnquiryStatus::factory()->create();
        $type = EnquiryType::factory()->create();

        $response = $this->putJson(route('forms.update', $form->id), [
            'success_message' => 'Foo Baz Bar',
            'initial_status_id' => $status->id,
            'type_id' => $type->id,
            'initial_assignee' => 10,
            'actions' => [
                'send-copy-via-mail' => [
                    'enabled' => true,
                ]
            ],
        ]);

        $response->assertStatus(422);
        $this->assertFalse($form->actions()->where('action_name', 'send-copy-via-mail')->exists());
    }

    public function testUpdateFormWillRemoveDisabledActions()
    {
        $this->asUser();
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $status = EnquiryStatus::factory()->create();
        $type = EnquiryType::factory()->create();
        $form->actions()->create([
            'action_name' => 'foo',
            'fields' => '[]',
        ]);

        $response = $this->putJson(route('forms.update', $form->id), [
            'success_message' => 'Foo Baz Bar',
            'initial_status_id' => $status->id,
            'type_id' => $type->id,
            'initial_assignee' => 10,
            'actions' => [
                'send-copy-via-mail' => [
                    'enabled' => true,
                    'recipient' => 'john@example.com',
                    'enquiry_subject' => 'Lorem ipsum',
                    'enquiry_content' => 'foo baz bar'
                ]
            ],
        ]);

        $response->assertStatus(200);
        $this->assertFalse($form->actions()->where('action_name', 'foo')->exists());
        $this->assertTrue($form->actions()->where('action_name', 'send-copy-via-mail')->exists());
    }

    public function testShowFormWithAction()
    {
        $this->asUser();
        $form = Form::factory()->create([
            'name' => 'foo',
            'is_active' => 1,
        ]);

        $form->actions()->create([
            'action_name' => 'foo',
            'fields' => [
                'foo' => 1,
                'baz' => 2
            ],
        ]);

        $response = $this->getJson(route('forms.show', $form->id));

        $response->assertStatus(200);
        $this->assertArrayHasKey('actions', $response->json());
        $this->assertArrayHasKey('foo', $response->json('actions'));
        $this->assertEquals(1, $response->json('actions.foo.foo'));
        $this->assertEquals(2, $response->json('actions.foo.baz'));
    }
}
