<?php

namespace Mtc\Crm\Exports;

use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Mtc\Crm\Models\FormQuestion;
use Mtc\MercuryDataModels\Enquiry;

class EnquiryExport implements FromCollection, WithHeadings
{
    private Collection $questions;

    public function __construct(private readonly array $input)
    {
        $this->questions = $this->getQuestions();
    }

    /**
    * @return \Illuminate\Support\Collection
    */
    public function collection(): Collection
    {
        return Enquiry::query()
            ->with([
                'form',
                'reason'
            ])
            ->when(
                ($this->input['type'] ?? 'all') !== 'all',
                fn($query) => $query->where('form_id', $this->input['type'])
            )
            ->when(
                !empty($this->input['from']),
                fn($query) => $query->where('created_at', '>=', $this->input['from'])
            )
            ->when(
                !empty($this->input['to']),
                fn($query) => $query->where('created_at', '<=', $this->input['to'])
            )
            ->get()
            ->map(fn(Enquiry $enquiry) => $this->mapRow($enquiry));
    }

    private function mapRow(Enquiry $enquiry): array
    {
        $row = [
            $enquiry->form?->name,
            $enquiry->getAttribute('reason.name', true),
            $enquiry->email,
            $enquiry->created_at?->format('Y-m-d H:i'),
        ];

        foreach ($this->questions as $name) {
            $row[] = $enquiry->getAttribute('value.' . $name, true);
        }

        return $row;
    }

    public function headings(): array
    {
        return array_merge([
            'Type',
            'Related object',
            'Email',
            'Submitted date',
        ], $this->questions->toArray());
    }

    private function getQuestions()
    {
        return FormQuestion::query()
            ->when(
                ($this->input['type'] ?? 'all') !== 'all',
                fn($query) => $query->where('form_id', $this->input['type'])
            )
            ->whereHas('form')
            ->whereNotIn('type', ['message', 'submit_button', 'file_upload'])
            ->pluck('name')
            ->unique();
    }
}
