<?php

namespace Mtc\Crm\Jobs;

use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Models\Enquiry;

class ProcessIncomingEnquiryJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected Enquiry $enquiry)
    {
        //
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->process()) {
            $this->enquiry->update([
                'ingested_at' => Carbon::now()
            ]);
        }
    }

    private function process(): bool
    {
        if ($this->enquiry->actions()->exists() === false) {
            return true;
        }

        // mark as processed if at least one action was successful
        return collect($this->enquiry->actions)
            ->filter(fn(EnquiryActionModel $action) => $this->handleEnquiryAction($action))
            ->isNotEmpty();
    }

    private function handleEnquiryAction(EnquiryActionModel $action): bool
    {
        try {
            $handler = Enquiries::makeAction($action->action_name);
            if ($handler->handle($action, $this->enquiry) === false) {
                $action->update([
                    'failed_at' => Carbon::now(),
                    'context' => 'Failed with reason: ' . $handler->failureReason()
                ]);
                return false;
            }

            $action->update([
                'processed_at' => Carbon::now(),
                'context' => $handler->successDetails(),
            ]);
        } catch (Exception $exception) {
            Log::error('Enquiry action failed', [
                'enquiry' => $this->enquiry,
                'action' => $action,
                'error' => $exception,
            ]);
            $action->update([
                'failed_at' => Carbon::now(),
                'context' => 'Failed due to a system error'
            ]);
            return false;
        }

        return true;
    }
}
