<?php

namespace Mtc\Crm\Tests\Unit;

use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Queue;
use Mtc\Crm\Console\Commands\IngestEnquiries;
use Mtc\Crm\Jobs\ProcessIncomingEnquiryJob;
use Mtc\Crm\Mail\NewEnquiryCustomerCopyNotification;
use Mtc\Crm\Mail\NewEnquiryMail;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\Crm\Models\EnquiryStatus;
use Mtc\Crm\Models\EnquiryTag;
use Mtc\Crm\Models\EnquiryType;
use Mtc\Crm\Tests\AsUser;
use Mtc\Crm\Tests\TestCase;
use Mtc\Crm\Tests\User;

class ProcessEnquiryTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function test_no_actions()
    {
        $enquiry = Enquiry::factory()->create(['ingested_at' => null]);

        (new ProcessIncomingEnquiryJob($enquiry))->handle();
        $enquiry->refresh();
        $this->assertNotNull($enquiry->ingested_at);
    }

    public function test_good_send_email_action()
    {
        $enquiry = Enquiry::factory()->create(['ingested_at' => null]);
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'data' => [
                'recipient' => 'john@example.com',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => null,
            'failed_at' => null,
        ]);

        Mail::fake();
        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $action->refresh();
        $enquiry->refresh();
        $this->assertNotNull($enquiry->ingested_at);
        $this->assertNotNull($action->processed_at);
        $this->assertNull($action->failed_at);

        Mail::assertSent(NewEnquiryMail::class, function (NewEnquiryMail $mail) {
            return $mail->hasTo('john@example.com');
        });
    }

    public function test_bad_send_email_action()
    {
        $enquiry = Enquiry::factory()->create(['ingested_at' => null]);
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'data' => [
                'recipient' => '',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => null,
            'failed_at' => null,
        ]);

        Mail::fake();
        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $action->refresh();
        $enquiry->refresh();
        $this->assertNull($enquiry->ingested_at);
        $this->assertNull($action->processed_at);
        $this->assertNotNull($action->failed_at);

        Mail::assertSent(NewEnquiryCustomerCopyNotification::class);
        Mail::assertNotSent(NewEnquiryMail::class);
    }

    public function test_broken_action()
    {
        $enquiry = Enquiry::factory()->create(['ingested_at' => null]);
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'foo-baz-bar',
            'data' => [
                'recipient' => '',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => null,
            'failed_at' => null,
        ]);

        Mail::fake();
        (new ProcessIncomingEnquiryJob($enquiry))->handle();

        $action->refresh();
        $enquiry->refresh();
        $this->assertNull($enquiry->ingested_at);
        $this->assertNull($action->processed_at);
        $this->assertNotNull($action->failed_at);
        $this->assertEquals('Failed due to a system error', $action->context);

        Mail::assertSent(NewEnquiryCustomerCopyNotification::class);
        Mail::assertNotSent(NewEnquiryMail::class);
    }
}
