<?php

namespace Mtc\Crm\Jobs;

use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Contracts\EnquiryModel;

class ProcessIncomingEnquiryJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected EnquiryModel $enquiry)
    {
        $this->onQueue('emails');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->process()) {
            $this->enquiry->refresh()->update([
                'ingested_at' => Carbon::now()
            ]);
        }
    }

    private function process(): bool
    {
        $this->sendConfirmation();
        if ($this->enquiry->actions()->exists() === false) {
            return true;
        }

        // mark as processed if at least one action was successful
        return collect($this->enquiry->actions)
            ->filter(fn(EnquiryActionModel $action) => $this->handleEnquiryAction($action))
            ->isNotEmpty();
    }

    private function handleEnquiryAction(EnquiryActionModel $action): bool
    {
        try {
            $handler = Enquiries::makeAction($action->action_name);
            if ($handler->handle($action, $this->enquiry) === false) {
                $action->update([
                    'failed_at' => Carbon::now(),
                    'context' => 'Failed with reason: ' . $handler->failureReason()
                ]);
                return false;
            }

            $action->update([
                'processed_at' => Carbon::now(),
                'context' => $handler->successDetails(),
            ]);
        } catch (Exception $exception) {
            Log::error('Enquiry action failed', [
                'enquiry' => $this->enquiry,
                'action' => $action,
                'error' => $exception,
            ]);
            $action->update([
                'failed_at' => Carbon::now(),
                'context' => 'Failed due to a system error'
            ]);
            return false;
        }

        return true;
    }

    private function sendConfirmation(): void
    {
        if ($this->enquiry->confirmation_sent) {
            return;
        }

        try {
            $class = config('crm.enquiry_confirmation_mail');
            Mail::to($this->enquiry->email)->send(new $class($this->enquiry));
            $this->enquiry->update([
                'confirmation_sent' => true,
            ]);
        } catch (\Exception $exception) {
            Log::warning($exception->getMessage());
        }
    }
}
