<?php

namespace Mtc\Crm\Providers;

use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Route;
use Mtc\Crm\Console\Commands\IngestEnquiries;
use Mtc\Crm\Contracts\Customer;
use Mtc\Crm\Contracts\CustomerNote;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryMessageModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Contracts\EnquiryObjectModel;
use Mtc\Crm\Contracts\EnquiryStatusModel;
use Mtc\Crm\Contracts\EnquiryTagModel;
use Mtc\Crm\Contracts\EnquiryTypeModel;
use Mtc\Crm\Contracts\FormActionModel;
use Mtc\Crm\Contracts\FormModel;
use Mtc\Crm\Contracts\FormQuestionModel;
use Mtc\Crm\Contracts\FormSectionModel;

class CrmProvider extends RouteServiceProvider
{
    /**
     * Register the service
     */
    public function register(): void
    {
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/crm.php', 'crm');
        if (config('crm.auto_load_migrations')) {
            $this->loadMigrationsFrom(__DIR__ . '/../../database/migrations/');
        }

        parent::register();

        $this->app->bind(Customer::class, Config::get('crm.customer_model'));
        $this->app->bind(CustomerNote::class, Config::get('crm.customer_note_model'));
        $this->app->bind(FormModel::class, Config::get('crm.form_model'));
        $this->app->bind(FormActionModel::class, Config::get('crm.form_action_model'));
        $this->app->bind(FormQuestionModel::class, Config::get('crm.form_question_model'));
        $this->app->bind(FormSectionModel::class, Config::get('crm.form_section_model'));
        $this->app->bind(EnquiryModel::class, Config::get('crm.enquiry_model'));
        $this->app->bind(EnquiryActionModel::class, Config::get('crm.enquiry_action_model'));
        $this->app->bind(EnquiryObjectModel::class, Config::get('crm.enquiry_object_model'));
        $this->app->bind(EnquiryMessageModel::class, Config::get('crm.message_model'));
        $this->app->bind(EnquiryTypeModel::class, Config::get('crm.type_model'));
        $this->app->bind(EnquiryStatusModel::class, Config::get('crm.status_model'));
        $this->app->bind(EnquiryTagModel::class, Config::get('crm.tag_model'));
    }

    /**
     * Boot the service
     */
    public function boot(): void
    {
        $this->loadViewsFrom(dirname(__DIR__, 2) . '/resources/views', 'crm');
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'crm');

        $this->app->bind('enquiries', function () {
            $repository = Config::get('crm.enquiry_repository_class');
            return new $repository(config('crm', []), $this->app->make(EnquiryModel::class));
        });

        $this->app->bind('enquiry-data', Config::get('crm.enquiry_data_repository_class'));
        $this->app->bind('forms', Config::get('crm.form_repository_class'));

        $this->app->make(EnquiryModel::class)->observe(Config::get('crm.enquiry_observer'));

        Relation::morphMap([
            'enquiry' => Config::get('crm.type_model'),
        ]);

        if ($this->app->runningInConsole()) {
            $this->publishes([
                dirname(__DIR__, 2) . '/database/migrations' => $this->app->databasePath('migrations'),
            ], 'migrations');
        }

        $this->commands([
            IngestEnquiries::class,
        ]);
    }

    /**
     * Define the "api" routes
     *
     * These routes all receive session state, CSRF protection, etc.
     *
     * @return void
     */
    public function map(): void
    {
        if (config('crm.load_routes')) {
            Route::prefix('api')
                ->middleware('api')
                ->group(dirname(__DIR__, 2) . '/routes/api.php');
        }
    }
}
