<?php

namespace Mtc\Crm\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Config;
use Mtc\Crm\Factories\EnquiryAnswerFactory;

abstract class EnquiryAnswerModel extends Model
{
    use HasFactory;

    /**
     * @var string
     */
    protected $table = 'enquiry_answers';

    /**
     * @var string[]
     */
    protected $fillable = [
        'enquiry_id',
        'question_id',
        'question_name',
        'answer_value',
        'answer_array',
        'answer_type',
    ];

    /**
     * @var array<string, string>
     */
    protected $casts = [
        'answer_array' => 'array',
    ];

    /**
     * Model factory
     */
    protected static function newFactory(): EnquiryAnswerFactory
    {
        return EnquiryAnswerFactory::new();
    }

    /**
     * Relationship with enquiry
     */
    public function enquiry(): BelongsTo
    {
        return $this->belongsTo(Config::get('crm.enquiry_model'), 'enquiry_id');
    }

    /**
     * Relationship with form question
     */
    public function question(): BelongsTo
    {
        return $this->belongsTo(Config::get('crm.form_question_model'), 'question_id');
    }

    /**
     * Get the answer value (returns array if answer_type is array, otherwise string)
     */
    public function getValueAttribute(): mixed
    {
        if ($this->answer_type === 'array' || !empty($this->answer_array)) {
            return $this->answer_array;
        }

        return $this->answer_value;
    }

    /**
     * Scope to filter by question ID
     */
    public function scopeForQuestion(Builder $query, int $questionId): Builder
    {
        return $query->where('question_id', $questionId);
    }

    /**
     * Scope to search by answer value
     */
    public function scopeWithAnswerValue(Builder $query, string $value): Builder
    {
        return $query->where('answer_value', 'like', "%{$value}%");
    }

    /**
     * Scope to filter by answer type
     */
    public function scopeOfType(Builder $query, string $type): Builder
    {
        return $query->where('answer_type', $type);
    }
}
