<?php

namespace Mtc\Coupons\Repositories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Coupons\Charts\Coupon as CouponChart;
use Mtc\Coupons\Contracts\CouponRepositoryInterface;
use Mtc\ShippingManager\Facades\ShippingModifiers;

/**
 * Class CouponRepository
 * @package Mtc\Coupons\Repositories
 */
class CouponRepository implements CouponRepositoryInterface
{
    /**
     * @var Model
     */
    protected $coupon;

    /**
     * @var Model
     */
    protected $basket;

    /**
     * @var
     */
    protected $applicable_items;

    /**
     * Get Coupon Class
     *
     * @return mixed
     */
    public function getCouponClass()
    {
        return App::make('coupon');
    }

    /**
     * Get Coupons
     *
     * @param bool $paginate
     * @return mixed
     */
    public function getCoupons(bool $paginate = true)
    {
        $query = $this->getCouponClass()
            ->newQuery()
            ->search(request());

        return $paginate ? $query->latest()->paginate(config('coupons.admin.pagination')) : $query->get();
    }

    /**
     * Find a coupon by code
     *
     * @param string $code
     */
    public function findByCode(string $code)
    {
        $this->coupon = $this->getCouponClass()
            ->newQuery()
            ->active()
            ->with('restrictions')
            ->where('code', $code)
            ->first();
    }

    /**
     * Find a coupon by id
     *
     * @param int $id
     */
    public function findById(int $id)
    {
        $this->coupon = $this->getCouponClass()
            ->newQuery()
            ->with('restrictions')
            ->where('id', $id)
            ->first();
    }

    /**
     * Set coupon model
     *
     * @param $coupon
     */
    public function setModel($coupon)
    {
        $this->coupon = $coupon;
    }

    /**
     * Set Basket model
     *
     * @param $basket
     */
    public function setBasketModel($basket)
    {
        $this->basket = $basket;
    }

    /**
     * Fetch the current coupon model
     *
     * @return int|mixed|null
     */
    public function getId()
    {
        return $this->coupon ? $this->coupon->id : null;
    }

    /**
     * Fetch the current coupon model
     *
     * @return Model
     */
    public function getModel()
    {
        return $this->coupon ?: $this->getCouponClass();
    }

    /**
     * Get coupon type
     *
     * @return mixed|null
     */
    public function getType()
    {
        return $this->coupon->type ?? null;
    }

    /**
     * Get coupon sale restriction
     *
     * @return integer
     */
    public function getSaleRestriction() : int
    {
        return $this->coupon->sale_restriction ?? 0;
    }

    /**
     * Get coupon restrictions for whole basket
     * 
     * @return mixed
     */
    public function getBasketRestrictions()
    {
        return $this->coupon
            ->restrictions
            ->whereIn('discountable_type', collect(config('coupons.restrictions.basket'))->keys())
            ->groupBy('discountable_type');
    }

    /**
     * Get coupon restrictions for basket items
     *
     * @return mixed
     */
    public function getBasketItemsRestrictions()
    {
        return $this->coupon
            ->restrictions
            ->whereIn('discountable_type', collect(config('coupons.restrictions.items'))->keys())
            ->groupBy('discountable_type');
    }

    /**
     * Get Sale Restrictions
     *
     * @return mixed
     */
    public function getSaleRestrictions()
    {
        return collect(config('coupons.sale_restrictions'));
    }

    /**
     * Get Basic restrictions
     *
     * @return Collection
     */
    public function getBasicRestrictions() : Collection
    {
        return collect(config('coupons.basic_restrictions'))
            ->filter(function($basic_restriction) {
                return $basic_restriction['enabled'] === true;
            });
    }

    /**
     * Get Restrictions
     *
     * @return Collection
     */
    public function getRestrictions()
    {
        return collect(config('coupons.restrictions'));
    }

    /**
     * Get Coupon Types
     *
     * @return \Illuminate\Support\Collection
     */
    public function getTypes()
    {
        return collect(config('coupons.discount_types', []))
            ->map(function ($type) {
                return $type['description'];
            });
    }

    /**
     * Get the shipping modifiers
     *
     * @return array|\Illuminate\Config\Repository|mixed
     */
    public function getShippingModifiers()
    {
        // skip if shipping doesn't exist
        if (!class_exists(ShippingModifiers::class)) {
            return [];
        }

        $modifiers = config('coupons.shipping_modifiers');
        if (count($modifiers) > 0) {
            $modifiers = collect($modifiers)
                ->map(function ($modifier, $name) {
                    return __("coupons::coupons.shipping_modifiers.{$name}");
                })
                ->prepend('Normal Shipping', '')
                ->toArray();
        }
        return $modifiers;
    }

    /**
     * Get Basket Applicable items
     */
    public function getBasketApplicableItems()
    {
        if (!$this->basket) {
            return;
        }

        $this->applicable_items = $this->basket->items
            ->filter(function ($item) {
                //Check if item is in discountables list
                return $this->isAvailableForItem($item);
            });

        return $this->applicable_items;
    }

    /**
     * Add Applicable Items to Basket Coupon
     */
    public function addApplicableItemsToBasket()
    {
        if (!$this->basket) {
            return;
        }

        $this->basket
            ->discounts()
            ->where('discount_type', 'coupon')
            ->where('discount_id', $this->getId())
            ->update([
                'applicable_items' => $this->applicable_items ? $this->applicable_items->pluck('purchasable.node.id') : null
            ]);
    }

    /**
     * Get coupons usage chart
     *
     * @return CouponChart|null
     */
    public function getChart()
    {
        return config('coupons.chart.enabled') ? new CouponChart($this->getModel(), request()) : null;
    }

    /**
     * Update or Create Model from request
     *
     * @param $request
     * @param null $id
     */
    public function updateOrCreateModel($request, $id = null)
    {
        if ($id > 0) {
            $this->findById($id);
        }

        $validated = $request->validated();
        $this->coupon = $this->getModel();
        $this->coupon->fill($validated);

        $basic_restrictions = $this->getBasicRestrictions()
            ->keys()
            ->mapWithKeys(function($basic_restriction) use ($validated) {
                $array[$basic_restriction] = isset($validated[$basic_restriction]) ? $validated[$basic_restriction] : '0';
                return $array;
            });

        $this->coupon->basic_restrictions = $basic_restrictions;
        $this->coupon->valid_from = empty($validated['valid_from']) ? null : Carbon::createFromFormat('d/m/Y', $validated['valid_from']);
        $this->coupon->valid_to = empty($validated['valid_to']) ? null : Carbon::createFromFormat('d/m/Y', $validated['valid_to']);
        $this->coupon->save();
    }

    /**
     * Find if coupon type is enabled
     *
     * @return bool
     */
    public function isTypeEnabled() : bool
    {
        $discount_type_class = config('coupons.discount_types.' . $this->getType() . '.class', false);
        return !empty($discount_type_class);
    }

    /**
     * Check if coupon is available for basket
     *
     * @param $basket
     * @return bool
     */
    public function isAvailable($basket)
    {
        if (
            $this->failsValidateBasicRestrictions($basket) ||
            $this->failsValidateBasketRestrictions() ||
            $this->failsValidateItemsRestrictions($basket)
        ) {
            return false;
        }
        
        return true;
    }

    /**
     * Check if item is available for coupon
     *
     * @param $item
     * @return bool
     */
    public function isAvailableForItem($item) : bool
    {
        $failed_validators = collect(config('coupons.item_checks'))
            ->reject(function($validator_class) use ($item) {
                return App::make($validator_class)->validate($this->coupon, $item);
            })
            ->count();

        if ($failed_validators) {
            return false;
        }

        $coupon_restrictions = $this->coupon
            ->restrictions
            ->whereIn('discountable_type', collect(config('coupons.restrictions.items'))->keys());

        if ($coupon_restrictions->count() === 0) {
            return true;
        }

        $excluded_restrictions = $coupon_restrictions->where('exclude', true);
        $included_restrictions = $coupon_restrictions->where('exclude', false)
            ->groupBy('discountable_type');

        $restrictions_count = [
            'excluded' => $excluded_restrictions->count(),
            'included' => $included_restrictions->count()
        ];

        $passed_included_restrictions = $included_restrictions
            ->filter(function($restrictions, $discountable_type) use ($item, $restrictions_count) {
                return $restrictions
                    ->filter(function($restriction) use ($item, $restrictions_count) {
                        return App::make(config('coupons.restrictions.items.' . $restriction->discountable_type . '.validation'))
                            ->validate($this->coupon, $item, $restriction, $restrictions_count);
                    })
                    ->count();
            });

        $failed_included_restrictions = $included_restrictions->diffAssoc($passed_included_restrictions)
            ->map(function($restrictions, $discountable_type) {
                return config('coupons.restrictions.items.' . $discountable_type . '.priority', 0);
            })
            ->sortByDesc(function($priority) {
                return $priority;
            });

        $passed_included_restrictions = $passed_included_restrictions
            ->map(function($restrictions, $discountable_type) {
                return config('coupons.restrictions.items.' . $discountable_type . '.priority', 0);
            })
            ->sortByDesc(function($priority) {
                return $priority;
            });

        $failed_excluded_restrictions = $excluded_restrictions
            ->reject(function($restriction) use ($item, $restrictions_count) {
                return App::make(config('coupons.restrictions.items.' . $restriction->discountable_type . '.validation'))
                    ->validate($this->coupon, $item, $restriction, $restrictions_count);
            })
            ->map(function($restriction) {
                return config('coupons.restrictions.items.' . $restriction->discountable_type . '.priority', 0);
            })
            ->sortByDesc(function($priority) {
                return $priority;
            });

        $passed_excluded_restrictions = $excluded_restrictions->diffAssoc($failed_excluded_restrictions)
            ->map(function($restriction) {
                return config('coupons.restrictions.items.' . $restriction->discountable_type . '.priority', 0);
            })
            ->sortByDesc(function($priority) {
                return $priority;
            });

        if ($restrictions_count['excluded'] === 0) {
            if ($failed_included_restrictions->count() && $passed_included_restrictions->count()) {
                if ($failed_included_restrictions->first() > $passed_included_restrictions->first()) {
                    return true;
                }
            }
        }

        if ($failed_excluded_restrictions->count() === 0) {
            if ($passed_included_restrictions->count() && $passed_excluded_restrictions->count()) {
                if ($passed_included_restrictions->first() < $passed_excluded_restrictions->first()) {
                    return true;
                }
            }

            if ($failed_included_restrictions->count() && $passed_excluded_restrictions->count()) {
                if ($failed_included_restrictions->first() > $passed_excluded_restrictions->first()) {
                    return true;
                }
            }
        }

        if ($passed_included_restrictions->count() !== $restrictions_count['included']) {
            return false;
        }

        if ($failed_excluded_restrictions->count()) {
            if ($passed_included_restrictions->count()) {
                if ($failed_excluded_restrictions->first() > $passed_included_restrictions->first()) {
                    return false;
                }
            } else {
                return false;
            }
        }

        return true;
    }

    /**
     * Check if coupons passes all basic restrictions
     *
     * @return bool
     */
    public function failsValidateBasicRestrictions($basket) : bool
    {
        $failed_validators = collect(config('coupons.basic_restrictions'))
            ->filter(function($basic_restriction) {
                return $basic_restriction['enabled'] === true && !empty($basic_restriction['validator']);
            })
            ->pluck('validator')
            ->reject(function($validator_class) use ($basket) {
                return App::make($validator_class)->validate($this->coupon, $basket);
            })
            ->count();
        
        return (bool)$failed_validators;
    }

    /**
     * Check basket restrictionss
     *
     * @return bool
     */
    public function failsValidateBasketRestrictions() : bool
    {
        if(!$this->getBasketRestrictions()->count()) {
            return false;
        }

        return $this->getBasketRestrictions()
                ->reject(function($restrictions, $discountable_type) {
                    return App::make(config('coupons.restrictions.basket.' . $discountable_type . '.validation'))
                        ->validate($this->coupon, $restrictions);
                })
                ->count() > 0;
    }

    /**
     * Check basket items restrictions
     *
     * @return bool
     */
    public function failsValidateItemsRestrictions($basket) : bool
    {
        return $basket->items
                ->filter(function($item) {
                    return $this->isAvailableForItem($item);
                })
                ->count() === 0;
    }

    /**
     * Delete Model by Id
     *
     * @param $id
     */
    public function deleteById($id)
    {
        $this->findById($id);

        if ($this->getModel()) {
            $this->getModel()->delete();
        }
    }
}
