<?php

namespace Mtc\Coupons;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Basket;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;

/**
 * Class Coupon
 *
 * @package Mtc\Coupons
 *
 * @property Carbon valid_to
 * @property Carbon valid_from
 */
class Coupon extends Model
{
    use HasPrices;
    
    /**
     * Sale restriction constants
     */
    const ALL_ITEMS = 0;
    const NON_SALE_ITEMS = 1;
    const ONLY_SALE_ITEMS = 2;

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'template_id',
        'code',
        'name',
        'redemptions',
        'type',
        'value',
        'shipping_modifier',
        'sale_restriction',
        'basic_restrictions',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'valid_from' => 'datetime',
        'valid_to' => 'datetime',
        'is_template' => 'boolean',
        'basic_restrictions' => 'array',
        'sale_restriction' => 'integer'
    ];

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        self::creating(function(self $coupon) {
            $coupon->code = strtoupper($coupon->code);
        });

        self::deleting(function (self $coupon) {
            $coupon->restrictions()->delete();
        });
    }

    /**
     * Scope - active()
     * Discards all hidden and deleted items
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        $current_date = Carbon::today()->format('Y-m-d');
        return $query->where('redemptions', '>', 0)
            ->where(function ($from) use ($current_date) {
                return $from->where('valid_from', '<=', $current_date)
                    ->orWhereNull('valid_from');
            })
            ->where(function ($to) use ($current_date) {
                return $to->where('valid_to', '>=', $current_date)
                    ->orWhereNull('valid_to');
            });
    }

    /**
     * Scope - search()
     * Used for setting search params
     *
     * @param Builder $query
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request)
    {
        collect(config('coupons.admin_search_filters', []))
            ->each(function($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query;
    }

    /**
     * Define the relationship to coupon restrictions
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function restrictions()
    {
        return $this->hasMany(Restriction::class);
    }

    /**
     * Define the relationship to coupon codes
     *
     * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function template()
    {
        return $this->hasMany(Coupon::class, 'template_id');
    }

    /**
     * Check if code already exists
     *
     * @param string $code coupon code to check
     * @return bool whether this code already exists
     */
    public function exists($code)
    {
        return self::query()
            ->where('code', $code)
            ->exists();
    }

    /**
     * Fetch the coupon from its code
     *
     * @param string $code
     * @return Builder|Model|object|null
     */
    public function getFromCode(string $code)
    {
        return self::query()
            ->where('code', $code)
            ->first();
    }

    /**
     * Check if code is existing redeemable coupon code
     *
     * @param string $code
     * @return mixed
     */
    public function isRedeemable(string $code)
    {
        return self::query()
            ->where('code', $code)
            ->where(function ($template) {
                return $template->where('is_template', 0)
                    ->orWhereNull('is_template');
            })
            ->active()
            ->exists();
    }

    /**
     * Check if coupon is active
     *
     * @return string
     */
    public function getStatusAttribute() : string
    {
        if ($this->attributes['redemptions'] == 0) {
            return 'No redemptions left';
        }

        if ($this->valid_to && $this->valid_to < Carbon::now()) {
            return 'Expired ' . $this->valid_to->format('Y-m-d');
        }

        if ($this->valid_from && $this->valid_from >= Carbon::now()) {
            return 'Upcoming';
        }

        return 'Active';
    }


    /**
     * @return mixed
     * @throws \Exception
     */
    public function getAmountAttribute()
    {
        if (empty($this->price_models['amount'])) {

            $config = [
                'price_entered_with_tax' => config('basket.discount_display_with_vat', false)
            ];
            $price = new Price($this->applicableToBasket(App::make('basket')), null, $config);
            $price->calculate();
            $this->price_models['amount'] = $price;
        }
        return $this->price_models['amount'];
    }

    /**
     * Get the applicable amount on basket
     *
     * @param Basket $basket
     * @return mixed
     */
    public function applicableToBasket(Basket $basket)
    {
        $this->deductable_amount = $basket->getCostSubtotalAttribute();
        App::make(config('coupons.discount_types.' . $this->type . '.class'))->apply($this, $basket);
        return $this->discounted_amount;
    }

    /**
     * Get the name value
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        if (!empty($this->attributes['name'])) {
            return $this->attributes['name'];
        }

        return __('coupons::coupons.basket_name', [
            'code' => $this->attributes['code']
        ]);
    }

    /**
     * Coupons are removable from basket
     *
     * @return bool
     */
    public function getIsRemovableAttribute()
    {
        return true;
    }

    /**
     * Get Basic Restriction Value
     *
     * @param $restriction
     * @return |null
     */
    public function getBasicRestriction($restriction)
    {
        return isset($this->basic_restrictions[$restriction]) ? $this->basic_restrictions[$restriction] : null;
    }

    /**
     * Get Discounted values
     *
     * @param $basket_item
     * @return mixed
     */
    public function getDiscountedValues($basket_item)
    {
        return App::make(config('coupons.discount_types.' . $this->type . '.class'))->getDiscountedValues($this, $basket_item);
    }
}
