<?php

namespace Mtc\Coupons\Contracts;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Coupons\Charts\Coupon as CouponChart;

/**
 * Interface CouponRepositoryInterface
 * @package Mtc\Coupons\Contracts
 */
interface CouponRepositoryInterface
{
    /**
     * Get Coupon Class
     *
     * @return mixed
     */
    public function getCouponClass();

    /**
     * Get Coupons
     *
     * @param bool $paginate
     * @return mixed
     */
    public function getCoupons(bool $paginate = true);

    /**
     * Find a coupon by code
     *
     * @param string $code
     */
    public function findByCode(string $code);

    /**
     * Find a coupon by id
     *
     * @param int $id
     */
    public function findById(int $id);

    /**
     * Set coupon model
     *
     * @param $coupon
     */
    public function setModel($coupon);

    /**
     * Fetch the current coupon model
     *
     * @return int|mixed|null
     */
    public function getId();

    /**
     * Fetch the current coupon model
     *
     * @return Model
     */
    public function getModel();

    /**
     * Get coupon type
     *
     * @return mixed|null
     */
    public function getType();

    /**
     * Get coupon sale restriction
     *
     * @return integer
     */
    public function getSaleRestriction(): int;

    /**
     * Find if coupon type is enabled
     *
     * @return bool
     */
    public function isTypeEnabled(): bool;

    /**
     * Get coupon restrictions for whole basket
     *
     * @return mixed
     */
    public function getBasketRestrictions();

    /**
     * Get coupon restrictions for basket items
     *
     * @return mixed
     */
    public function getBasketItemsRestrictions();

    /**
     * Check if item is available for coupon
     *
     * @param $item
     * @return bool
     */
    public function isAvailableForItem($item): bool;

    /**
     * Get Sale Restrictions
     *
     * @return mixed
     */
    public function getSaleRestrictions();

    /**
     * Get Basic restrictions
     *
     * @return Collection
     */
    public function getBasicRestrictions(): Collection;

    /**
     * Get Restrictions
     *
     * @return \Illuminate\Support\Collection
     */
    public function getRestrictions();

    /**
     * Get Coupon Types
     *
     * @return \Illuminate\Support\Collection
     */
    public function getTypes();

    /**
     * Get the shipping modifiers
     *
     * @return array|\Illuminate\Config\Repository|mixed
     */
    public function getShippingModifiers();

    /**
     * Check if coupon is available for basket
     *
     * @param $basket
     * @return bool
     */
    public function isAvailable($basket);

    /**
     * Check if coupons passes all basic restrictions
     *
     * @return bool
     */
    public function failsValidateBasicRestrictions($basket): bool;

    /**
     * Check basket restrictionss
     *
     * @return bool
     */
    public function failsValidateBasketRestrictions(): bool;

    /**
     * Check basket items restrictions
     *
     * @return bool
     */
    public function failsValidateItemsRestrictions($basket): bool;

    /**
     * Get coupon usage chart
     *
     * @return CouponChart|null
     */
    public function getChart();

    /**
     * Update or Create Model from request
     *
     * @param $request
     * @param null $id
     */
    public function updateOrCreateModel($request, $id = null);

    /**
     * Delete Model by Id
     *
     * @param $id
     */
    public function deleteById($id);
}
