<?php

namespace Mtc\Coupons\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Coupons\Contracts\CouponRepositoryInterface;
use Mtc\Coupons\Facades\Coupon;
use Mtc\Coupons\Http\Requests\CreateCouponRequest;
use Mtc\Coupons\Http\Requests\UpdateCouponRequest;
use Mtc\ShippingManager\Facades\ShippingModifiers;

/**
 * Class CouponController
 *
 * @package Mtc\Coupons\Http\Controllers\Admin
 */
class CouponController extends Controller
{
    /**
     * @var CouponRepositoryInterface
     */
    protected $coupon_repository;

    /**
     * CouponController constructor.
     * @param CouponRepositoryInterface $coupon_repository
     */
    public function __construct(CouponRepositoryInterface $coupon_repository)
    {
        $this->coupon_repository = $coupon_repository;
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = 'Manage Coupons';
        return template('admin/coupons/index.twig', [
            'page_meta' => $this->page_meta,
            'coupons' => $this->coupon_repository->getCoupons(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Coupon';
        return template('admin/coupons/edit.twig', [
            'coupon' => $this->coupon_repository->getCouponClass(),
            'sale_restrictions' => $this->coupon_repository->getSaleRestrictions(),
            'page_meta' => $this->page_meta,
            'coupon_types' => $this->coupon_repository->getTypes(),
            'delivery_options' => $this->coupon_repository->getShippingModifiers(),
            'form_action' => route('coupons.store')
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateCouponRequest $request
     * @return RedirectResponse
     */
    public function store(CreateCouponRequest $request): RedirectResponse
    {
        $this->coupon_repository->updateOrCreateModel($request);
        session()->flash('success', 'Coupon Created');
        return redirect()->route('coupons.edit', [ $this->coupon_repository->getId() ]);
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Request $request, int $id)
    {
        $this->coupon_repository->findById($id);
        $this->page_meta['title'] = 'Edit Coupon';
        return template('admin/coupons/edit.twig', [
            'restrictions' => $this->coupon_repository->getRestrictions(),
            'basic_restrictions' => $this->coupon_repository->getBasicRestrictions(),
            'sale_restrictions' => $this->coupon_repository->getSaleRestrictions(),
            'coupon' => $this->coupon_repository->getModel(),
            'coupon_types' => $this->coupon_repository->getTypes(),
            'delivery_options' => $this->coupon_repository->getShippingModifiers(),
            'page_meta' => $this->page_meta,
            'chart' => $this->coupon_repository->getChart(),
            'form_action' => route('coupons.update', [ $id ])
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param UpdateCouponRequest $request
     * @param int $id
     * @return RedirectResponse
     */
    public function update(UpdateCouponRequest $request, int $id): RedirectResponse
    {
        $this->coupon_repository->updateOrCreateModel($request, $id);
        session()->flash('success', 'Coupon Updated');
        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return RedirectResponse
     */
    public function destroy(int $id): RedirectResponse
    {
        $this->coupon_repository->deleteById($id);
        session()->flash('success', 'Coupon Deleted');
        return redirect()->back();
    }
}
