<?php

namespace Mtc\Core\Models;

use Mtc\Core\Models\Country;

/**
 * Core CountryState Class
 * Class for country states or other equivalent country sub regions
 * @author Rihards Silins
 * @version 1 13/10/2016
 * @copyright MTC media Ltd
 */
class CountryState extends \Illuminate\Database\Eloquent\Model
{
    /**
     * Indicates if the model should be timestamped.
     * @var bool
     */
    public $timestamps = false;

    /**
     * Belongs to relationshop to country model
     * @return IIlluminate\Database\Eloquent\Relations\BelongsTo Object $relationship
     */
    public function country()
    {
        return $this->belongsTo("Mtc\Core\Models\Country");
    }

    /**
     * Getting array of states with codes as keys and names as values contained within country code array
     * @param string $country_code optional
     * @param bool $include_countries_without_states optional
     * @return string[]|string[][]
     * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
     */
    public static function getCountryStateList($country_code = null, $include_countries_without_states = false)
    {
        if (empty($country_code)) {
            $country_list = Country::getCountryList();
        } else {
            $country_list = [
                $country_code => $country_code
            ];
        }
        $country_and_their_states_list = [];
        foreach ($country_list as $country_code => $temp_value) {
            $states = self::getStateList($country_code);

            if (empty($states) && $include_countries_without_states === true) {
                $country_and_their_states_list[$country_code] = [];
            } elseif (!empty($states)) {
                $country_and_their_states_list[$country_code] = $states;
            }
        }

        return $country_and_their_states_list;
    }

    /**
     * Getting array of states with codes as keys and names as values
     * @param string $country_code
     * @return string[]
     * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
     */
    public static function getStateList($country_code)
    {
        return self::whereHas('country', function ($query) use ($country_code) {
            $query->where('code', '=', $country_code);
        })
            ->select('code', 'name')
            ->orderBy('name', 'asc')
            ->pluck('name', 'code')
            ->all();
    }

    /**
     * Get state name from code
     * The most common way states are retrieved since codes are used as values and names as labels
     * @param string $code_state code
     * @param string $state_code code
     * @return string
     * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
     */
    public static function getNameByCode($country_code, $state_code)
    {
        $state = self::whereHas('country', function ($query) use ($country_code) {
            $query->where('code', '=', $country_code);
        })
            ->where("code", $state_code)
            ->first(["name"]);
        if ($state == null) {
            return false;
        }
        return $state->name;
    }
}
