<?php
namespace Mtc\Core\Http\Controllers\Auth;

use Mtc\Core\Rules\SecurePassword;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Illuminate\Auth\Passwords\TokenRepositoryInterface;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Auth\ResetsPasswords;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Mtc\Core\Admin\User;
use Mtc\Core\Http\Controllers\Controller;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;

/**
 * Class ResetPasswordController
 *
 * @package Mtc\Core\Http\Controllers\Auth
 */
class ResetPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset emails and
    | includes a trait which assists in sending these notifications from
    | your application to your users. Feel free to explore this trait.
    |
    */

    use ResetsPasswords;

    /**
     * @var Application
     */
    protected $app;

    /**
     * @var TokenRepositoryInterface
     */
    protected $tokens;

    /**
     * Get the password reset validation rules.
     *
     * @return array
     */
    protected function rules()
    {
        return [
            'token' => 'required',
            'email' => 'required|email',
            'password' => [
                'required',
                'confirmed',
                'min:6',
                new SecurePassword()
            ]
        ];
    }

    /**
     * Render the password reset form
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Request $request)
    {
        if (!$request->hasValidSignature()) {
            return template('admin/auth/reset_password.twig', [
                'token_error' => 'The password token is not valid'
            ]);
        }

        return template('admin/auth/reset_password.twig', [
            'token' => $request->token,
            'email' => $request->email
        ]);
    }

    /**
     * Reset the given user's password.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function reset(Request $request)
    {
        $this->validate($request, $this->rules(), $this->validationErrorMessages());

        /** @var User $user */
        $user = User::query()->where('email', $request->input('email'))->first();
        if (!$user) {
            session()->flash('errors', [ 'password' => 'Admin ccount was not found']);
            return redirect()->back();
        }

        $user = $this->validateReset($request, $user);
        if (! $user instanceof CanResetPasswordContract) {
            return $user;
        }

        $password = $request->password;
        $this->resetPassword($user, $password);
        $this->tokens->delete($user);

        return redirect()->to(route('admin.login'));
    }

    /**
     * Validate the token is correct
     * @param $credentials
     * @param $user
     * @return mixed
     */
    protected function validateReset($credentials, $user)
    {
        $this->app = app();
        $config = $this->app['config']["auth.passwords.users"];
        $this->tokens = $this->createTokenRepository($config);

        if (! $this->tokens->exists($user, $credentials['token'])) {
            return Password::INVALID_TOKEN;
        }

        return $user;
    }

    /**
     * Create token repository
     *
     * @param array $config
     * @return DatabaseTokenRepository
     */
    protected function createTokenRepository(array $config)
    {
        $key = $this->app['config']['app.key'];

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $connection = $config['connection'] ?? null;

        return new DatabaseTokenRepository(
            $this->app['db']->connection($connection),
            $this->app['hash'],
            $config['table'],
            $key,
            $config['expire']
        );
    }
    /**
     * Reset the given user's password.
     *
     * @param  \Illuminate\Contracts\Auth\CanResetPassword  $user
     * @param  string  $password
     * @return void
     */
    protected function resetPassword($user, $password)
    {
        $user->password = Hash::make($password);
        $user->save();

        event(new PasswordReset($user));
    }

}