<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Seo\Canonical as SeoCanonical;
use \DB;

/**
 * Manages the installation of modules and plugins within the Core admin.
 */
class SeoCanonicalController extends Controller
{

    /**
     * Displays the list of Canonicals taken from SeoCanonicals table
     *
     * @return mixed
     */
    public function index()
    {
        $this->page_meta['title'] = 'Manage SEO Canonical URLs';
        $canonicals = SeoCanonical::get();

        return template('admin/seo/canonicals.twig', [
            'page_meta' => $this->page_meta,
            'canonicals' => $canonicals,
        ]);
    }

    /**
     * Validates the form data before storing it in SeoCanonical table
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'path' => 'required',
            'canonical_path' => 'required',
        ]);

        $canonical = SeoCanonical::firstOrCreate([
            'path' => $request->input('path'),
            'canonical_path' => $request->input('canonical_path'),
        ]);
        $canonical->save();

        session()->flash('success', 'Your canonical path has now been added!');
        return redirect()->back();
    }

    /**
     * Validates the form data before updating the SeoCanonical table
     *
     * @param Request $request
     * @param id $canonical_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $canonical_id)
    {
        $this->validate($request, [
            'path' => 'required',
            'canonical_path' => 'required'
        ]);

        SeoCanonical::where('id', $canonical_id)
            ->update([
                'path' => $request->input('path'),
                'canonical_path' => $request->input('canonical_path')
            ]);

        session()->flash('success', 'Your canonical has now been updated!');
        return redirect()->back();
    }

    /**
     * Displays the form with data depending on selected Canonical
     *
     * @param int $canonical_id
     * @return mixed
     */
    public function edit($canonical_id)
    {
        $this->page_meta['title'] = 'Manage SEO Canonical URLs';
        $canonical = SeoCanonical::where('id', $canonical_id)->get();

        return template('admin/seo/edit_canonicals.twig', [
            'page_meta' => $this->page_meta,
            'canonical' => $canonical,
        ]);
    }

    /**
     * Deletes the Canonical which has been selected
     *
     * @param int $canonical_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($canonical_id)
    {
        $delete_canonical = SeoCanonical::find($canonical_id);
        $delete_canonical->delete();

        session()->flash('success', 'Your canonical has now been deleted!');
        return redirect()->back();
    }
}

