<?php

namespace Mtc\Core\Console\Commands;

use Illuminate\Console\Command;
use Mtc\Core\Images\ImageOptimise;
use Spatie\ImageOptimizer\OptimizerChain;
use Spatie\ImageOptimizer\Optimizers\Jpegoptim;
use Spatie\ImageOptimizer\Optimizers\Optipng;
use Spatie\ImageOptimizer\Optimizers\Pngquant;

/**
 * Class ImageOptimiseCommand
 *
 * @package App\Console\Commands
 */
class ImageOptimiseCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'images:optimise {--lossy}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimise images in uploads folder';

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle()
    {
        $image_paths = [
            base_path('/uploads/images/*/*.png'),
            base_path('/uploads/images/*/*.jpg'),
            base_path('/uploads/images/*/*/*.png'),
            base_path('/uploads/images/*/*/*.jpg'),
        ];

        foreach ($image_paths as $path) {
            foreach (glob($path) as $image) {
                if (strpos($image, 'original')) {
                    continue;
                }
                $this->info("Optimizing {$image}");

                // Jpegoptim is used for compressing jpeg/jpg files
                $jpeg_optim = new Jpegoptim([
                    '--strip-all',
                    '--all-progressive',
                    // maintain 95% size. Going below this
                    (!$this->option('lossy') ?: '--max=80')
                ]);
                $jpeg_optim->binaryName = '/usr/local/bin/jpegoptim';

                // PNGQuant is used to compress image (lossy compression)
                $png_quant = new Pngquant([
                    '--force',
                ]);
                $png_quant->binaryName = '/usr/local/bin/pngquant';

                // Optipng is a losless compression method
                $opti_png = new Optipng([
                    '-i0',
                    '-o2',
                    '-quiet',
                ]);
                $opti_png->binaryName = '/usr/local/bin/optipng';

                // Create and execute optimizer chain on image
                $optimizer_chain = (new OptimizerChain())
                    ->addOptimizer($jpeg_optim)
                    ->addOptimizer($png_quant)
                    ->addOptimizer($opti_png);
                $optimizer_chain->optimize($image);

            }
        }

        $this->info("Done !");
    }
}
