<?php

namespace Mtc\Core\Console\Commands;

use DirectoryIterator;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\App;

/**
 * Class ReoploadCommand
 *
 * @package Mtc\Core\Console\Commands
 */
class ReuploadCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'images:reupload {image_folder_key} {--single_size=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Reprocesses already uploaded images, useful when $image_folders are changed and images need to be recropped according to new rules.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $folders = App::make('image_folders');

        $images_key = $this->argument('image_folder_key');

        // array key we expect to find originals under
        $originals_key = 'originals';

        $folders = $folders[$images_key];

        // Second argument - the 'Only size' to resize
        if ($only_size = $this->option('single_size')) {
            if (!isset($folders[$only_size])) {
                $this->error("Size [$only_size] not found");
                exit();
            }

            // reduce the $folders to only 'originals' and the 'only size'
            $folders = [
                $originals_key => $folders[$originals_key],
                $only_size => $folders[$only_size],
            ];
        }

        $uploader = \Mtc\Core\Images\ImageUploader::newInstance($folders);
        // set rename to false to keep existing filenames
        $uploader->rename = false;

        // assume that original images are found under 'originals' key
        if (!isset($folders[$originals_key])) {
            $this->error("'originals' key not found in the folders array");
            exit();
        }

        // iterate the files in the originals folder and simulate an image upload
        $dir = new DirectoryIterator(
            base_path('/' . trim($folders[$originals_key]['path'], '/'))
        );

        foreach ($dir as $file) {
            if ($file->isDot() || $file->isDir()) {
                $this->info("Skipping {$file->getFilename()}");
                continue;
            }

            $file = new \Symfony\Component\HttpFoundation\File\UploadedFile(
                $file->getPathname(),
                $file->getFilename()
            );

            $this->line("Reuploading {$file->getFilename()}");

            $filedata = $uploader->uploadImage($file);

            if (is_array($filedata)) {
                $this->line("Uploaded to {$filedata['name']}");
                if ($filedata['error']) {
                    $this->error("ERROR: {$filedata['error']}");
                }
            }
        }
    }
}
