<?php

namespace Mtc\Core;

/**
 * Class FormField
 *
 * Form Field generator function.
 * Allows generating base input fields
 *
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class FormField
{
    /**
     * Constants for defining a layout
     */
    const LAYOUT_DEFAULT = 0;
    const LAYOUT_ADMIN_FILTER = 1;

    /**
     * Allowed layout paths
     *
     * @var array
     */
    private static $layout_paths = [
        self::LAYOUT_DEFAULT => 'includes/input_fields',
        self::LAYOUT_ADMIN_FILTER => 'includes/input_fields/admin_filter',
        'admin' => 'includes/input_fields/admin_filter',
    ];

    /**
     * @var string[] currently supported fields and which templates they use
     */
    private static $supported_fields = [
        'text' => 'text',
        'email' => 'text',
        'tel' => 'text',
        'password' => 'text',
        'number' => 'text',
        'textarea' => 'textarea',
        'radio' => 'radio',
        'select' => 'select',
        'checkbox' => 'checkbox',
        'image' => 'image',
        'captcha' => 'captcha',
    ];

    /**
     * Render a form field based on given $options.
     * This ensures all form fields are inline with general mtc styling.
     * Expected fields in options:
     * - type
     * - value
     * - id
     * - label
     * - errors (array)
     * - class (optional)
     * - field_value (checkbox|radio)
     * - values (select)
     *
     * @param string[] $options list of rendering params
     * @param \Twig_Environment $twig
     * @return string|void html of input field|nothing if template not supported
     */
    public static function renderField(Array $options, \Twig_Environment $twig)
    {
        if (empty($options['type'])) {
            return;
        }

        if (!array_key_exists($options['type'], self::$supported_fields)) {
            return;
        }

        $layout = self::getLayout($options);
        $template = self::$supported_fields[$options['type']];
        return $twig->render("{$layout}/{$template}.twig", $options);
    }

    /**
     * Find the layout used for the field
     *
     * @param $options
     * @return mixed
     * @throws \Exception
     */
    protected static function getLayout($options)
    {
        if (empty ($options['layout'])) {
            return self::$layout_paths[self::LAYOUT_DEFAULT];
        }

        if (!array_key_exists($options['layout'], self::$layout_paths)) {
            throw new \Exception('Unsupported layout');
        }

        return self::$layout_paths[$options['layout']];
    }
}
