<?php
/**
 * Core Country Class
 * Class for countries
 * @author Rihards Silins
 * @author Vladislavs Ignatjevs <vladislavs.ignatjevs@mtcmedia.co.uk>
 * @version 2 23/09/2016
 * @copyright MTC media Ltd
 */

namespace Mtc\Core\Models;

use Mtc\Core\Currency;
use GeoIp2\Database\Reader;

/**
 * Country
 *
 * Select countries from their table
 *
 * @author Rihards Silins
 * @author Vladislavs Ignatjevs <vladislavs.ignatjevs@mtcmedia.co.uk>
 *
 * version 2 18/10/2016
 */

class Country extends \Illuminate\Database\Eloquent\Model
{
    public $timestamps = false;

    /**
     * The table associated with a model
     *
     * @var string
     */
    protected $table = 'countries';

    /**
     * Detect Location
     *
     * @return array Details of country detected
     */
    public function detectLocation($request, $cookie)
    {
        // Initialise empty so we always return an array
        $location = [];

        // Check if country already selected
        if (!empty($cookie['detected_country'])) {
            $_SESSION['detected_country'] = $cookie['detected_country'];
        }

        // 216.37.72.238 is a US ip address for testing
        if (is_file('/usr/local/lib/php/GeoIP/GeoLite2-Country.mmdb')
            && empty($_SESSION['detected_country']))
        {
            /*
             * Create reader object
             */
            if (!empty($_SERVER["REMOTE_ADDR"])) {
                $reader = new Reader('/usr/local/lib/php/GeoIP/GeoLite2-Country.mmdb');
                $country = $reader->country($_SERVER["REMOTE_ADDR"]);
            }

            // If we have a country code
            if (!empty($country->country->isoCode)) {
                $_SESSION['detected_country'] = $country->country->isoCode;
                setcookie('detected_country', $country->country->isoCode);
            }
        }

        if (!empty($_SESSION['detected_country'])) {

            $location = self::query()
                ->where('code', $_SESSION['detected_country'])
                ->where('status', 1)
                ->first();

            if (!empty($location)) {
                $location = $location->toArray();
            }

            $_SESSION['location'] = $location;

        }

        return $location;

    }

    /**
     *
     * Defining if the country is from Europe by country code or country name
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @param string country
     * @return boolean
     *
     */
    public static function isEurope($country)
    {
        return Country::where('continent_code', '=', 'EU')
                      ->where(function ($query) use ($country) {
                          $query->where('name', '=', $country)
                                ->orWhere('code', '=', $country)
                          ;
                      })
                      ->count() > 0;
    }

    /**
     *
     * Defining if the country is from European Union by country code or country name
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @param string country
     *
     * @return boolean
     */
    public static function isEu($country)
    {
        return Country::where('is_eu', '=', '1')
                      ->where(function ($query) use ($country) {
                          $query->where('name', '=', $country)
                                ->orWhere('code', '=', $country)
                          ;
                      })
                      ->count() > 0;
    }

    /**
     *
     * Returns countries that are members of European Union with
     * codes as keys and names as values
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @return string[]
     */
    public static function getEuCountryList()
    {
        return Country::where('is_eu', '=', '1')
                      ->orderBy('code', 'asc')
                      ->pluck('name', 'code')
                      ->all();
    }

    /**
     *
     * Getting list of all countries (full name) with
     * codes as keys and names as values
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @return string[]
     *
     */
    static function getFullCountryList()
    {
        return static::getCountryList();
    }


    /**
     *
     * Getting array of europe countries with codes as keys and names as values
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @return string[]
     *
     */
    static function getEuropeCountryList()
    {
        return static::getCountryList('EU');
    }

    /**
     *
     * Getting array of countries with codes as keys and names as values
     *
     * @author Vladislavs Ignatjevs <vladislavs.igntatjevs@mtcmedia.co.uk>
     *
     * @param string $continent_code
     * @return string[]
     *
     */
    public static function getCountryList($continent_code = '')
    {
        return Country::select('code', 'name')
                      ->where(function ($query) use ($continent_code) {
                          if (!empty($continent_code)) {
                              $query->where('continent_code', '=', $continent_code);
                          }
                      })
                      ->orderBy('order', 'asc')
                      ->orderBy('name', 'asc')
                      ->pluck('name', 'code')
                      ->all();
    }

    /**
     * Get country name from code
     * The most common way countries are retrieved since codes are used as values and names as labels
     * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
     * @param string $code country code
     * @return string/false $name country name or false if code matches no country
     */
    public static function getNameByCode($code) {
        $country = Country::where("code", $code)
            ->first(["name"]);
        if ($country == null) {
            return false;
        }

        return $country->name;
    }

    /**
     *
     * Defining if the country uses postcodes in its' mail system
     * example input:
     * $country = 'Germany'
     * or
     * $country = 'DE'
     *
     * @author Jack Donaldson <jack.donaldson@mtcmedia.co.uk>
     *
     * @param string $country

     * @return boolean
     *

     *
     */
    public static function hasPostcodes($country)
    {
        return Country::where('has_postcodes', '=', '1')
                      ->where(function ($query) use ($country) {
                          $query->where('name', '=', $country)
                                ->orWhere('code', '=', $country);
                      })
                      ->count() > 0;
    }

    /**
     *
     * Retrieves the Regular Expression for the inputted country's postcode
     *
     * If no result is found, the code will assume that the country doesn't require a postcode
     *
     * @param string $country_code Expected input is the country code : "DR", "FR", "US", etc
     * @return string | null Expected output would be similar to "[0-9]{5}(-\s)[0-9]{4}" if $country = "US"
     *
     */
    public static function getRegex($country_code)
    {
        return self::where("code", $country_code)->value("regex");
    }
}
