<?php

namespace Mtc\Core\Traits;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Core\Node;

/**
 * Trait to allow for the connection between a Model and corresponding Node.
 *
 * @category Mtc\Core
 * @package Mtc\Core
 * @author Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
trait Nodeable
{
    /**
     * Define the relationship to the node.
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphOne
     */
    public function node()
    {
        return $this->morphOne(Node::class, 'nodeable');
    }

    /**
     * Load the controller for this object
     *
     * @throws \Exception
     */
    public function loadController(Request $request)
    {
        $controller_name = $this->controller_name ?: $this->defaultController();
        $controller_method = $this->controller_method ?: 'show';

        if (method_exists(new $controller_name, $controller_method)) {
            $controller = App::make($controller_name);
            return App::call([$controller, $controller_method], ['request' => $request]);
        }
        throw new \BadMethodCallException("Method '{$controller_method}' not found for {$controller_name}");
    }

    /**
     * Magically load the controller from the known namespace
     *
     * @return string
     * @throws \ReflectionException
     * @throws \Exception
     */
    protected function defaultController()
    {
        $controller_name = class_basename($this) . 'Controller';
        $controller_namespace = (new \ReflectionClass(get_class($this)))->getNamespaceName();
        $controller_class_name = $controller_namespace . '\\Http\\Controllers\\' . $controller_name;
        if (class_exists($controller_class_name)) {
            return $controller_class_name;
        }
        throw new \BadMethodCallException("Controller {$controller_name} not found for route");
    }

    /**
     * Check whether wild-card routes are allowed
     *
     * @return bool
     */
    public function allowPartialMatch()
    {
        return !empty($this->allow_partial_url_match);
    }
}
