<?php

namespace Mtc\AnalyticsStats;

use Google\Analytics\Data\V1beta\BetaAnalyticsDataClient;
use Google\Analytics\Data\V1beta\Filter\StringFilter;
use Google\Analytics\Data\V1beta\RunReportResponse;
use Google\Analytics\Data\V1beta\FilterExpression;
use Google\Analytics\Data\V1beta\RunReportRequest;
use Google\Analytics\Data\V1beta\MetricType;
use Google\Analytics\Data\V1beta\DateRange;
use Google\Analytics\Data\V1beta\Dimension;
use Google\Analytics\Data\V1beta\Filter;
use Google\Analytics\Data\V1beta\Metric;
use Carbon\Carbon;

class Stats
{
    /**
     * @var \BetaAnalyticsDataClient
     */
    protected $analytics;

    /**
     * @var string
     */
    protected $property_id;

    /**
     * @param $property_id
     */
    public function __construct($property_id)
    {
        $this->analytics = new BetaAnalyticsDataClient([
            'credentials' => dirname(__DIR__) . '/client_secrets.json',
        ]);

        $this->property_id = (string)$property_id;
    }

    /**
     * @param $startDate
     * @param $endDate
     * @param array $metrics
     * @param array $dimensionFilter
     * @param string $dimension
     *
     * @return array
     * @throws \Exception
     */
    public function getReport($startDate, $endDate, $metrics = [], $dimensionFilter = [], $dimension = 'date')
    {
        $start = $startDate === '' ? Carbon::now() : Carbon::parse($startDate);
        $end = $endDate === '' ? Carbon::now() : Carbon::parse($endDate);
        $analytics_data = [];

        try {
            $request = [
                'property' => 'properties/' . $this->property_id,
                'dateRanges' => [
                    new DateRange([
                        'start_date' => $start->format('Y-m-d'),
                        'end_date'   => $end->format('Y-m-d'),
                    ]),
                ],
                'dimensions' => [
                    new Dimension(['name' => $dimension]),
                ],
                'metrics' => collect($metrics)->map(function ($metric) {
                    return new Metric(['name' => $metric]);
                })->toArray()
            ];

            if (!empty($dimensionFilter)) {
                $request['dimensionFilter'] = $dimensionFilter;
            }

            $response = $this->analytics->runReport($request);

            if (!empty($response->getRows())) {
                foreach ($response->getRows() as $row) {
                    $rowData = [];
                    $metricValues = $row->getMetricValues();
                    $dimensionValue = $row->getDimensionValues()[0]->getValue();

                    foreach ($metricValues as $index => $value) {
                        $rowData[$metrics[$index]] = $value->getValue();
                    }

                    $rowData['date'] = $dimensionValue;
                    $analytics_data[] = $rowData;
                }
            }

            return $analytics_data;
        } catch (\Exception $exception) {
            throw new \RuntimeException($exception->getMessage());
        }
    }

    /**
     * @param $startDate
     * @param $endDate
     *
     * @return int
     * @throws \Exception
     */
    public function getTotalVisitors($startDate = '', $endDate = '')
    {
        $metric = 'activeUsers';
        $report = $this->getReport($startDate, $endDate, [$metric], []);
        $totalSessions = array_sum(array_column($report, $metric));

        return max($totalSessions, 0);
    }

    /**
     * @throws \Exception
     */
    public function getTotalVisitorsToPage($startDate = '', $endDate = '', $pagePath = '')
    {
        $metric = 'screenPageViews';
        $filter = [];

        if ($pagePath !== '') {
            $filter = new FilterExpression([
                'filter' => new Filter([
                    'field_name' => 'pagePath',
                    'string_filter' => new StringFilter([
                        'match_type' => Filter\StringFilter\MatchType::CONTAINS,
                        'value' => $pagePath,
                    ])
                ])
            ]);
        }

        $report = $this->getReport($startDate, $endDate, [$metric], $filter, 'pagePath');
        $totalSessions = array_sum(array_column($report, $metric));

        return max($totalSessions, 0);
    }

    /**
     * @param $startDate
     * @param $endDate
     *
     * @return mixed
     * @throws \Exception
     */
    public function getTotalVisits($startDate = '', $endDate = '')
    {
        $metric = 'screenPageViews';
        $report = $this->getReport($startDate, $endDate, [$metric], []);
        $totalSessions = array_sum(array_column($report, $metric));

        return max($totalSessions, 0);
    }

    /**
     * @param $startDate
     * @param $endDate
     * @param callable $getTotalOrderCallback - returns total orders during the specified period
     *
     * @return float|int
     */
    public function getConversionRate($startDate = null, $endDate = null, callable $getTotalOrderCallback = null)
    {
        $total = 0;
        $visitors = $this->getTotalVisitors($startDate, $endDate);

        if (is_callable($getTotalOrderCallback)) {
            $total = $getTotalOrderCallback($startDate, $endDate);
        }

        if (!$visitors) {
            return 0;
        }

        return ($total / $visitors) * 100;
    }

    /**
     * @param $startDate
     * @param $endDate
     * @param $metrics
     *
     * @return array
     * @throws \Exception
     */
    public function getStats($startDate = '', $endDate = '', $metrics = [])
    {
        if (empty($metrics)) {
            $metrics = [
                'sessions',
                'screenPageViews',
                'screenPageViewsPerUser',
                'activeUsers',
                'totalUsers',
                'transactions'
            ];
        }

        $report = $this->getReport($startDate, $endDate, $metrics, [], 'date');

        $report_data = [];

        foreach ($metrics as $metric) {
            $report_data[$metric] = array_sum(array_column($report, $metric));
        }

        return $report_data;
    }
}