<?php

namespace Mtc\AnalyticsStats;

use Carbon\Carbon;

class Stats
{
    /** @var \Google_Client */
    private $client;

    /** @var \Google_Service_AnalyticsReporting */
    protected $analytics;
    protected $view_id;

    public function __construct($view_id)
    {
        $this->view_id = (string)$view_id;
        $this->client = new \Google_Client();
        $this->client->setAuthConfig(dirname(__DIR__) . '/client_secrets.json');
        $this->client->addScope(\Google_Service_Analytics::ANALYTICS_READONLY);
        $this->analytics = new \Google_Service_AnalyticsReporting($this->client);
    }

    public function getReport($startDate, $endDate, \Google_Service_AnalyticsReporting_Metric $metric, $filter = null)
    {
        $start = $startDate == '' ? Carbon::now() : Carbon::parse($startDate);
        $end = $endDate == '' ? Carbon::now() : Carbon::parse($endDate);

        // Create the DateRange object.
        $date_range = new \Google_Service_AnalyticsReporting_DateRange();
        $date_range->setStartDate($start->format('Y-m-d'));
        $date_range->setEndDate($end->format('Y-m-d'));

        // Create the ReportRequest object.
        $request = new \Google_Service_AnalyticsReporting_ReportRequest();
        $request->setViewId($this->view_id);
        $request->setDateRanges($date_range);
        $request->setMetrics([$metric]);

        if ($filter !== null) {
            $clause = new \Google_Service_AnalyticsReporting_DimensionFilterClause();
            $clause->setFilters([ $filter ]);
            $request->setDimensionFilterClauses([ $clause ]);
        }

        $body = new \Google_Service_AnalyticsReporting_GetReportsRequest();
        $body->setReportRequests([$request]);
        try {
            $response = $this->analytics->reports->batchGet($body);
            return $response
                ->getReports()[0]
                ->getData()
                ->getTotals()[0]
                ->getValues()[0];
        } catch (\Exception $exception) {
            return 0;
        }
    }

    public function getTotalVisitors($startDate = '', $endDate = '')
    {
        // Create the Metrics object.
        $visitors = new \Google_Service_AnalyticsReporting_Metric();
        $visitors->setExpression("ga:visitors");
        $visitors->setAlias("visitors");

        return $this->getReport($startDate, $endDate, $visitors);
    }

    public function getTotalVisitorsToPage($startDate = '', $endDate = '', $page = '')
    {
        // Create the Metrics object.
        $metric = new \Google_Service_AnalyticsReporting_Metric();
        $metric->setExpression("ga:pageviews");
        $metric->setAlias("pageviews");

        $filter = new \Google_Service_AnalyticsReporting_DimensionFilter();
        $filter->setOperator('EXACT');
        $filter->setDimensionName('ga:pagePath');
        $filter->setExpressions([ $page ]);

        return $this->getReport($startDate, $endDate, $metric, $filter);
    }

    public function getTotalVisits($startDate = '', $endDate = '', $page = '')
    {
        // Create the Metrics object.
        $visits = new \Google_Service_AnalyticsReporting_Metric();
        $visits->setExpression("ga:visits");
        $visits->setAlias("visits");

        return $this->getReport($startDate, $endDate, $visits);
    }

    /**
     * @param callable $getTotalOrderCallback - returns total orders during the specified period
     */
    public function getConversionRate(string $startDate = null, string $endDate = null, callable $getTotalOrderCallback = null): int
    {
        $total = 0;
        $visitors = $this->getTotalVisitors($startDate, $endDate);

        if(is_callable($getTotalOrderCallback)) {
            $total = call_user_func($getTotalOrderCallback, $startDate, $endDate);
        }

        if (!$visitors) {
            return 0;
        }

        return ($total / $visitors) * 100;
    }
}