<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Contracts\Validation\Factory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\Media;
use Mtc\ContentManager\Tests\TestCase;

class MediaModelTest extends TestCase
{
    public function testUsesRelationship(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        self::assertInstanceOf(HasMany::class, $model->uses());
    }

    public function testResizableAttribute(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        $tests = [
            'image' => true,
            'audio' => false,
            'video' => false,
        ];
        foreach ($tests as $test => $expectation) {
            $model->type = $test;
            self::assertEquals($expectation, $model->getResizableAttribute());
        }

    }

    public function testSquareThumbAttribute(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        $model->path = '/2025-Apr/orig/';
        $model->src = 'abc.jpg';
        $model->type = 'image';
        app('config')->set('media.default_thumbnail_size', [
            'width' => 200,
            'height' => 200,
        ]);

        self::assertStringContainsString('200x200/abc.jpg', $model->getSquareThumbUrlAttribute());
    }


    public function testNonResizableMimeDoesNotHaveSizeInUrl(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        $model->path = '/2025-Apr/orig/';
        $model->src = 'abc.jpg';
        $model->type = 'svg';
        app('config')->set('media.default_thumbnail_size', [
            'width' => 200,
            'height' => 200,
        ]);

        self::assertFalse(Str::contains($model->getSquareThumbUrlAttribute(), '200x200'));
    }

    public function testOriginalPathAttribute(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        $model->src = 'abc.jpg';
        $model->path = 'something';

        self::assertStringContainsString('something/abc.jpg', $model->getOriginalFilePath());
    }

    public function testOriginalUrlAttribute(): void
    {
        /** @var Media $model */
        $model = app(Media::class);
        $model->src = 'abc.jpg';
        $model->path = 'something';

        self::assertStringContainsString('something/abc.jpg', $model->getOriginalUrlAttribute());

        /** @var \Illuminate\Contracts\Validation\Factory $validator */
        $validator = app(Factory::class);
        $wrong_url = $validator->make([
            'url' => $model->getOriginalUrlAttribute()
        ], [
            'url' => 'url'
        ])->fails();

        self::assertFalse($wrong_url);
    }
}
