<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Mtc\ContentManager\Contracts\Media as MediaModel;
use Mtc\ContentManager\Contracts\MediaFolder;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Http\Requests\AddImagesToFolderRequest;
use Mtc\ContentManager\Http\Requests\BulkDeleteFoldersRequest;
use Mtc\ContentManager\Http\Requests\BulkDeleteMediaRequest;
use Mtc\ContentManager\Http\Requests\BulkUpdateFoldersRequest;
use Mtc\ContentManager\Http\Requests\ReorderFoldersRequest;
use Mtc\ContentManager\Http\Requests\StoreFolderRequest;
use Mtc\ContentManager\Http\Resources\MediaFileUsageResource;

class MediaLibraryController
{
    use ValidatesRequests;

    /**
     * Show media library home.
     * This shows all details like types, dates and available tags
     *
     * @param Request $request
     * @return array
     */
    public function index(Request $request)
    {
        return [
            'files' => Media::getFiles($request),
            'types' => Media::getTypes(),
            'upload_dates' => Media::getUploadDates(),
            'tags' => Media::getTags(),
            'folders' => Media::getFolders($request)
        ];
    }

    /**
     * Show media files. Allows filtering by request.
     *
     * @param Request $request
     * @return array
     */
    public function show(Request $request)
    {
        return Media::getFiles($request);
    }

    /**
     * Show use cases of media file
     *
     * @param MediaModel $media
     * @return MediaFileUsageResource
     */
    public function showUses(MediaModel $media)
    {
        return new MediaFileUsageResource($media->uses);
    }

    /**
     * Remove the media object
     *
     * @param int $mediaId
     * @return Response
     */
    public function bulkDelete(BulkDeleteMediaRequest $request): Response
    {
        Media::destroyMedia($request->input('id'));
        return response('success', 200);
    }

    public function storeFolder(StoreFolderRequest $request): MediaFolder
    {
        return Media::createFolder($request->input('name'), $request->input('parent_id'));
    }

    public function bulkUpdateFolders(BulkUpdateFoldersRequest $request): Response
    {
        $folders = $request->input('folders');

        app(MediaFolder::class)::rebuildTree($folders);

        return response('success', 200);
    }

    /**
     *  Delete the media folders
     *
     * @param BulkDeleteFoldersRequest $request
     * @return array
     */
    public function bulkDeleteFolders(BulkDeleteFoldersRequest $request): array
    {
        $folders = app(MediaFolder::class)
            ->whereIn('id', $request->input('id'))
            ->get();

        foreach ($folders as $folder) {
            Media::deleteFolder($folder);
        }

        return [
            'folders' => Media::getFolders($request)
        ];
    }

    /**
     * Reorder media folders
     *
     * @param ReorderFoldersRequest $request
     * @return Response
     */
    public function reorderFolders(ReorderFoldersRequest $request): Response
    {
        Media::reorderFolders($request->input('folders'));

        return response('success', 200);
    }

    /**
     * Add images to folder
     *
     * @param AddImagesToFolderRequest $request
     * @return Response
     */
    public function addImages(AddImagesToFolderRequest $request): Response
    {
        Media::addImagesToFolder($request->input('folder'), $request->input('images'));

        return response('success', 200);
    }
}
