<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use JetBrains\PhpStorm\ArrayShape;

interface ElementRepositoryContract
{
    /**
     * Attach model to repository
     *
     * @param ContentElement $element
     * @return $this
     */
    public function setModel(ContentElement $element): self;

    /**
     * Find the element
     *
     * @param int $id
     * @return ContentElement|Model
     */
    public function find(int $id): ContentElement;

    /**
     * Load element by id for usage
     *
     * @param int $id
     * @return $this
     */
    public function load(int $id): self;

    /**
     * Get list
     *
     * @param string|null $searchTerm
     * @param string|null $sortBy
     * @param int $limit
     * @param bool $active
     * @param bool $paginate
     * @return LengthAwarePaginator|Collection
     */
    public function getList(
        string $searchTerm = null,
        string $sortBy = null,
        int $limit = 10,
        bool $active = true,
        bool $paginate = true
    ): LengthAwarePaginator|Collection;

    /**
     * Create a new content element
     *
     * @param string $title
     * @param string $subtitle
     * @param string $icon
     * @param string $category
     * @param bool $enabled
     * @return ContentElement|Model
     */
    public function create(
        string $title,
        string $subtitle = null,
        string $icon = null,
        string $category = null,
        bool $enabled = true,
    ): ContentElement;

    public function update(ContentElement $element, array $input): ContentElement;

    public function makeCopy(int $elementId, string $title): ContentElement;

    /**
     * Add field to content element
     *
     * @param array $field
     * @return $this
     */
    #[ArrayShape([
        'child_element_id' => "int",
        'type' => "string",
        'name' => "string",
        'data' => "array",
        'choices' => "array",
    ])]
    public function addField(array $field): self;

    /**
     * Remove a content element
     *
     * @param int $elementId
     * @return bool
     */
    public function destroy(int $elementId): bool;

    /**
     * Get the list of places where content element is used
     *
     * @param int $elementId
     * @return array
     */
    public function getUsage(int $elementId): array;

    /**
     * Show full usage list of a specific content type
     *
     * @param int $elementId
     * @param string $type
     * @return Collection
     */
    public function fullUsageList(int $elementId, string $type): Collection;

    /**
     * Find Template records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findTemplatesUsingElement(int $elementId, int $limit = 10): Collection;

    /**
     * Find free-form pages (no template) records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findPagesUsingElement(int $elementId, int $limit = 10): Collection;

    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findGlobalElementsUsingElement(int $elementId, int $limit = 10): Collection;

    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findMenusUsingElement(int $elementId, int $limit = 10): Collection;
}
