<?php

namespace Mtc\ContentManager;

use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\Contracts\TemplateElement;

class TemplateRepository
{
    public const LAYOUT_TYPES = [
        'list',
        'columns',
    ];

    public function update(Request $request, Template $template, TemplateElement $templateElement)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $template->fill($request->input())->save();
        $template->elements()
            ->whereNotIn('id', collect($request->input('elements'))->pluck('id'))
            ->delete();

        collect($request->input('elements'))
            ->each(fn($element, $order) => $this->saveTemplateElement($element, $order, $template, $templateElement));

        if (config('pages.use_transactions')) {
            DB::commit();
        }
    }

    public function canRemove(Template $template): bool
    {
        return ($template->meta['disallow_delete'] ?? false) !== true;
    }

    public function contentToLayout(array $contentStructure): array
    {
        return collect($contentStructure)
            ->map(function ($block) {
                if (isset($block['type']) && in_array($block['type'], self::LAYOUT_TYPES)) {
                    $block['items'] = $this->contentToLayout($block['items']);
                } else {
                    $block['fields'] =  collect($block['fields'])->forget('value');
                }
                return $block;
            })
            ->toArray();
    }

    public function extractBlockIds(array $request_structure): array
    {
        return $this->recursiveExtractBlockIds($request_structure)
            ->flatten()
            ->unique()
            ->toArray();
    }

    protected function recursiveExtractBlockIds(array $requestStructure): Collection
    {
        return collect($requestStructure)
            ->map(function ($block) {
                if (isset($block['type']) && in_array($block['type'], self::LAYOUT_TYPES)) {
                    return $this->recursiveExtractBlockIds($block['items']);
                }
                return $block['block_id'];
            });
    }

    protected function saveTemplateElement(
        array $element,
        int $order,
        Template $template,
        TemplateElement $templateElement
    ): void {
        /** @var TemplateElement $data */
        if (empty($element['id']) || !is_numeric($element['id'])) {
            $element['id'] = null;
        }
        $data = $templateElement->newQuery()->findOrNew($element['id'] ?? 0);
        if ($data->exists === false) {
            $data->template_id = $template->id;
        }

        $data->fill([
            'data' => $this->fieldDataValues($element),
            'meta' => $element['meta'] ?? [],
            'name' => $element['name'],
            'field_type' => $element['field_type'],
            'global_content_id' => $element['global_content_id'] ?? null,
            'description' => $element['description'] ?? '',
            'hint' => $element['hint'] ?? '',
            'is_list' => $element['is_list'] ?? false,
            'max_repeat_count' => $element['max_repeat_count'] ?? false,
            'order' => $order,
            'slug' => $element['slug'] ?? null,
        ]);
        if (empty($element['global_content_id'])) {
            $data->element_id = $element['element_id'] ?? null;
        }

        $data->save();
    }

    protected function fieldDataValues($field): array
    {
        return collect($field)->except([
            'is_list',
            'max_repeat_count',
            'meta',
            'id',
            'name',
            'content',
            'children',
            'field_type',
            'order',
            'template_id',
            'element_id',
            'global_content_id',
        ])->toArray();
    }
}
