<?php

namespace Mtc\ContentManager\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Mtc\ContentManager\Contracts\Comment;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;

class NewCommentNotification extends Notification
{
    use Queueable;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct(protected Comment $comment)
    {
        //
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return MailMessage
     */
    public function toMail($notifiable)
    {

        switch ($this->getType()) {
            case 'page':
                return $this->pageComment();
            case 'template':
                return $this->templateComment();
            case 'page-reply':
                return $this->pageComment();
            case 'template-reply':
                return $this->templateComment();
        }
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            //
        ];
    }

    protected function getType(): string
    {
        $suffix = '';
        if ($this->comment->parent_id > 0) {
            $suffix = '-reply';
        }

        return $this->comment->commentable_type === 'template'
            ? 'template' . $suffix
            : 'page' . $suffix;
    }

    protected function pageComment()
    {
        $url = UiUrlGenerator::make('content/pages/' . $this->comment->commentable_id);

        return (new MailMessage())
            ->subject('A new comment has been added to ' . $this->comment->commentable->title)
            ->line(
                $this->comment->author->name . ' added a comment on page **'
                . $this->comment->commentable->title . '**'
            )
            ->line($this->comment->content)
            ->action('View Comment', $url);
    }

    protected function templateComment()
    {
        $url = UiUrlGenerator::make('structure/templates/' . $this->comment->commentable_id);

        return (new MailMessage())
            ->subject('A new comment has been added to ' . $this->comment->commentable->name)
            ->line(
                $this->comment->author->name . ' added a comment on page **'
                . $this->comment->commentable->name . '**'
            )
            ->line($this->comment->content)
            ->action('View Comment', $url);
    }

    protected function pageReplyComment()
    {
        $url = UiUrlGenerator::make('content/pages/' . $this->comment->commentable_id);

        return (new MailMessage())
            ->subject('A comment has been replied to on ' . $this->comment->commentable->title)
            ->line(
                $this->comment->author->name . ' replied to your comment on page **'
                . $this->comment->commentable->title . '**'
            )
            ->line($this->comment->content)
            ->action('View Comment', $url);
    }

    protected function templateReplyComment()
    {
        $url = UiUrlGenerator::make('structure/templates/' . $this->comment->commentable_id);
        return (new MailMessage())
            ->subject('A comment has been replied to on ' . $this->comment->commentable->name)
            ->line(
                $this->comment->author->name . ' replied to your comment on page **'
                . $this->comment->commentable->name . '**'
            )
            ->line($this->comment->content)
            ->action('View Comment', $url);
    }
}
