<?php

namespace Mtc\ContentManager;

use Carbon\Carbon;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\ImageManager;
use Mtc\ContentManager\Contracts\Media;
use Mtc\ContentManager\Contracts\MediaFolder;
use Mtc\ContentManager\Contracts\MediaTag;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Filters\FocalCropFilter;
use Mtc\ContentManager\Filters\WidthResizeFilter;
use Mtc\ContentManager\Http\Requests\MediaResizeRequest;
use Mtc\ContentManager\Http\Requests\MediaUploadRequest;
use Mtc\ContentManager\Jobs\ChangeSizesToMediaFocalPoint;
use Mtc\ContentManager\Jobs\GenerateSizeForMediaFileJob;
use Mtc\ContentManager\Jobs\RegenerateSizesForMediaFileJob;
use Mtc\ContentManager\Models\MediaSize;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleOffer;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class MediaRepository
{
    use DispatchesJobs;

    public static array $backup_image_formats = [
        'avif',
        'tiff',
        'jp2',
    ];

    public function __construct(
        protected Media $model,
        protected MediaUse $mediaUse,
        protected MediaTag $tag,
        protected MediaFolder $folder,
    ) {
        //
    }

    /**
     * List of media files
     *
     * @param Request $request
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function getList(Request $request): \Illuminate\Contracts\Pagination\LengthAwarePaginator
    {
        return $this->model->newQuery()
            ->when($request->has('model'), fn($query) => $query->where('parent_type', $request->input('model')))
            ->when($request->has('model_id'), fn($query) => $query->where('parent_id', $request->input('model_id')))
            ->when($request->has('folder'), function ($query) use ($request) {
                if ($request->input('folder') == 'uncategorised') {
                    $query->whereNull('folder_id');
                } else {
                    $query->where('folder_id', $request->input('folder'));
                }
            })
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters', []))
            ->paginate($request->input('per_page'));
    }

    /**
     * Find media file
     *
     * @param int $id
     * @return Media|Model
     */
    public function find(int $id): Media
    {
        return $this->model->newQuery()
            ->findOrFail($id);
    }

    /**
     * Import image from an external URL
     *
     * @param string $url
     * @param string $model
     * @return Media
     * @throws \Exception
     */
    public function importImageFromUrl(string $url, string $model = '', ?string $imageProvider = null): Media
    {
        $path = ltrim(parse_url($url)['path'], '/');
        $originalFileName = pathinfo($path, PATHINFO_BASENAME);
        $date = date('Y-M');
        $extension = pathinfo($originalFileName, PATHINFO_EXTENSION);
        $fileName = str_replace('.' . $extension, '', $originalFileName)
            . '-' . Str::random(5)
            . '-' . Carbon::now()->format('U');
        $uploadPath = $this->storagePrefix() . ltrim("$model/$date/orig/", '/')
            . substr($originalFileName, 0, 2) . '/' . substr($originalFileName, 0, 4);

        if (strlen($fileName) > 75) {
            $fileName = Str::random(75) . '.webp';
        } else {
            $fileName = Str::slug(Str::limit($fileName, 75)) . '.webp';
        }

        $fileNameJPG = str_replace('.webp', '.jpg', $fileName);

        $fileContents = @file_get_contents($url);
        if ($fileContents === false) {
            throw new FileNotFoundException('Unable to load file');
        }
        $image = $this->getImageManager($extension)->make($fileContents);

        if (empty(pathinfo($fileName, PATHINFO_EXTENSION))) {
            $fileName .= '.webp';
        }

        Storage::disk(Config::get('filesystems.default_media'))
            ->put(
                rtrim($uploadPath, '/') . '/' . $fileName,
                $image->stream('webp'),
                ['visibility' => 'public']
            );

        // store a copy in jpg for better handling with 3rd parties that do not support .webp
        Storage::disk(config('filesystems.default_media'))
            ->put(
                rtrim($uploadPath, '/') . '/' . $fileNameJPG,
                $image->stream('jpg'),
                ['visibility' => 'public']
            );

        $hexImg = $image->resize(10, 10);
        $hex1 = $hexImg->pickColor(0, 0, 'hex');
        $hex2 = $hexImg->pickColor(9, 9, 'hex');

        $media =  $this->createMediaRecordForFile(
            $uploadPath,
            $fileName,
            $image->mime(),
            true,
            null,
            $url,
            $imageProvider
        );

        $media->update([
            'hex1' => $hex1,
            'hex2' => $hex2,
        ]);

        return $media;
    }


    public function importImageFromFile(SplFileInfo $file, ?string $imageProvider = null): Media
    {
        $date = date('Y-M');
        $originalFileName = $file->getFilename();
        $extension = pathinfo($originalFileName, PATHINFO_EXTENSION);
        $fileName = str_replace('.' . $extension, '', $originalFileName)
            . '-' . Str::random(5)
            . '-' . Carbon::now()->format('U');
        $uploadPath = $this->storagePrefix() . ltrim("$date/orig/", '/')
            . substr($originalFileName, 0, 2) . '/' . substr($originalFileName, 0, 4);

        if (strlen($fileName) > 75) {
            $fileName = Str::random(75) . '.webp';
        } else {
            $fileName = Str::slug(Str::limit($fileName, 75)) . '.webp';
        }

        $filePath = rtrim($uploadPath, '/') . '/' .  $fileName;
        $filePathJpg = rtrim($uploadPath, '/') . '/' .  str_replace('.webp', '.jpg', $fileName);
        $image = $this->getImageManager($extension)->make($file->getContents());

        Storage::disk(Config::get('filesystems.default_media'))
            ->put($filePath, $image->stream('webp'), ['visibility' => 'public']);

        // store a copy in jpg for better handling with 3rd parties that do not support .webp
        Storage::disk(config('filesystems.default_media'))
            ->put($filePathJpg, $image->stream('jpg'), ['visibility' => 'public']);

        $hexImg = $image->resize(10, 10);
        $hex1 = $hexImg->pickColor(0, 0, 'hex');
        $hex2 = $hexImg->pickColor(9, 9, 'hex');

        $media =  $this->createMediaRecordForFile(
            $uploadPath,
            $fileName,
            $image->mime(),
            true,
            null,
            $originalFileName,
            $imageProvider
        );

        $media->update([
            'hex1' => $hex1,
            'hex2' => $hex2,
        ]);

        return $media;
    }


    /**
     * Handle new image upload
     *
     * @param Request $request
     * @param Media|null $mediaToUpdate
     * @return Media
     * @throws \Exception
     */
    public function upload(MediaUploadRequest $request, Media $mediaToUpdate = null): Media
    {
        if ($request->file('file')->getClientOriginalExtension() === 'zip') {
            return $this->uploadZip($request);
        }

        return $request->fileIsImage()
            ? $this->uploadImage($request, $mediaToUpdate)
            : $this->uploadFile($request, $mediaToUpdate);
    }

    /**
     * Handle new image upload
     *
     * @param Request $request
     * @return Media
     * @throws \Exception
     */
    public function uploadImage(
        MediaUploadRequest $request,
        Media $mediaToUpdate = null,
        string $localFilePath = null,
    ): Media {
        $file = $localFilePath && file_exists($localFilePath)
            ? new UploadedFile($localFilePath, basename($localFilePath))
            : $request->file('file');

        $timestamp = Carbon::now()->format('U');
        $fileName =  $file->getClientOriginalName();
        $uploadPath = $this->storagePrefix() . ltrim($request->input('model', '') . '/' . date('Y-M') . '/orig/', '/')
            . substr($fileName, 0, 2) . '/' . substr($fileName, 0, 4);
        $extension = pathinfo($fileName, PATHINFO_EXTENSION);
        $fileNameJPG = Str::slug(str_replace('.' . $extension, '', $fileName)) . $timestamp . '.jpg';
        $fileName = Str::slug(str_replace('.' . $extension, '', $fileName)) . $timestamp . '.webp';

        $image = $this->getImageManager($extension)
            ->make($file->getPathname())
            ->orientate();

        // With very large images there starts an imagick cache issue that is hard to configure about
        // Due to this we simply limit the max image size for the original to 5000px width
        if ($image->width() > 5000) {
            $aspect_ratio = $image->width() / $image->height();
            $image->resize(5000, 5000 / $aspect_ratio);
        }

        Storage::disk(config('filesystems.default_media'))
            ->put(
                $uploadPath . '/' . $fileName,
                $image->stream('webp'),
                ['visibility' => 'public']
            );

        // store a copy in jpg for better handling with 3rd parties that do not support .webp
        Storage::disk(config('filesystems.default_media'))
            ->put(
                $uploadPath . '/' . $fileNameJPG,
                $image->stream('jpg'),
                ['visibility' => 'public']
            );

        $hexImg = $image->resize(10, 10);
        $hex1 = $hexImg->pickColor(0, 0, 'hex');
        $hex2 = $hexImg->pickColor(9, 9, 'hex');

        $media =  $mediaToUpdate
            ? $this->updateRecordWithFile(
                $mediaToUpdate,
                $uploadPath,
                $fileName,
                $file->getMimeType(),
                true,
                $request->user()?->id
            )
            : $this->createMediaRecordForFile(
                $uploadPath,
                $fileName,
                $file->getMimeType(),
                true,
                $request->user()?->id,
            );

        $media->update([
            'hex1' => $hex1,
            'hex2' => $hex2,
        ]);

        return $media;
    }

    /**
     * Handle new file upload
     *
     * @param Request $request
     * @return Media
     * @throws \Exception
     */
    public function uploadFile(MediaUploadRequest $request, Media $mediaToUpdate = null): Media
    {
        $uploadPath = $this->storagePrefix() . ltrim($request->input('model') . '/' . date('Y-M'), '/');
        $fileName = Carbon::now()->format('U') . $request->file('file')->getClientOriginalName();

        Storage::disk(config('filesystems.default_media'))
            ->putFileAs(
                $uploadPath,
                $request->file('file'),
                $fileName,
                ['visibility' => 'public']
            );

        return $mediaToUpdate
            ? $this->updateRecordWithFile(
                $mediaToUpdate,
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                false,
                $request->user()?->id,
            )
            : $this->createMediaRecordForFile(
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                false,
                $request->user()?->id,
            );
    }

    /**
     * Attach media use cases to model
     *
     * @param int[] $mediaIds
     * @param Model $model
     * @param array $meta
     * @param bool $removeOthers
     * @return void
     */
    public function setUsesForModel(array $mediaIds, Model $model, array $meta = [], bool $removeOthers = true): void
    {
        $allowed_sizes = $this->getAllowedSizes($model, $meta);

        if ($removeOthers) {
            $this->removeMediaUsesNotInList($model, $mediaIds);
        }

        $this->model->newQuery()
            ->whereIn('id', $mediaIds)
            ->get()
            ->each(fn(Media $media) => $this->updateMediaMeta($media, $meta))
            ->map(fn(Media $media) => $this->setUsesForMedia(
                $media,
                array_search($media->id, $mediaIds),
                $allowed_sizes,
                $meta,
                $model
            ))
            ->filter(fn(MediaUse $use) => $use->wasRecentlyCreated ||  $use->wasChanged(['allowed_sizes']))
            ->each(fn(MediaUse $use) => $this->triggerSizeGenerationForMedia($use->media, $allowed_sizes, $model));

        $this->setLastUsedOnMedia($mediaIds);
        if (empty($media->folder_id)) {
            $this->setFolder($model, $mediaIds);
        }
    }


    public function removeMediaUsesNotInList(Model $model, array $mediaIds): void
    {
        $this->mediaUse->newQuery()
            ->whereNotIn('media_id', $mediaIds)
            ->where('owner_type', $model->getMorphClass())
            ->where('owner_id', $model->id)
            ->delete();
    }


    /**
     * Update media use order for model
     *
     * @param int[] $media
     * @param Model $model
     * @return void
     */
    public function setUseOrdering(array $mediaOrder, Model $model)
    {
        $data = collect($mediaOrder)
            ->map(fn($order, $id) => [
                'media_id' => $id,
                'order' => $order,
                'owner_id' => $model->id,
                'owner_type' => $model->getMorphClass()
            ])
            ->toArray();
        $this->mediaUse->newQuery()
            ->upsert($data, ['media_id', 'owner_id', 'owner_type'], ['order']);
    }

    /**
     * Get list of files in library
     *
     * @param Request $request
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function getFiles(Request $request)
    {
        return $this->model->newQuery()
            ->when($request->input('type'), fn($query) => $query->where('type', $request->input('type')))
            ->when($request->input('tag'), function ($query) use ($request) {
                return $query->whereHas('tags', fn($tag_query) => $tag_query->where('tag', $request->input('tag')));
            })
            ->when($request->input('upload_date'), function ($query) use ($request) {
                return $query->where('upload_date', $request->input('upload_date'));
            })
            ->paginate(Config::get('media.library_page_size'));
    }

    public function getUploadDates()
    {
        return $this->model->newQuery()
            ->select('upload_date')
            ->distinct()
            ->orderByDesc('upload_date')
            ->pluck('upload_date', 'upload_date')
            ->map(fn($upload_date) => Carbon::createFromFormat('Y-m', $upload_date)->format('F Y'));
    }

    public function getTypes()
    {
        return array_keys(Config::get('media.file_types', []));
    }

    /**
     * Get list of tags used
     *
     * @return Collection
     */
    public function getTags()
    {
        return $this->tag->newQuery()
            ->distinct()
            ->select('tag')
            ->get();
    }

    /**
     * Coordinates are defined as x1, y1, x2, y2
     *
     * @param MediaResizeRequest $request
     * @param array $coordinates
     * @return string
     * @throws \Exception
     */
    public function resize(MediaResizeRequest $request, array $coordinates): string
    {
        $cropWidth = $coordinates[2] - $coordinates[0];
        $cropHeight = $coordinates[3] - $coordinates[1];
        $size = new ImageSize($request->width(), $request->height());
        $original = $request->media()->media;

        $file = Storage::disk(Config::get('filesystems.default_media'))->get($original->getOriginalFilePath());
        $image = $this->getImageManager('webp')
            ->make($file)
            ->crop($cropWidth, $cropHeight, $coordinates[0], $coordinates[1])
            ->resize($size->getWidth(), $size->getHeight())
            ->fit($size->getWidth(), $size->getHeight());

        Storage::disk(Config::get('filesystems.default_media'))
            ->put(
                $size->fileDestination($original->src, $original->path, $original->legacy_structure),
                $image->stream()
            );

        return Storage::disk(Config::get('filesystems.default_media'))
            ->url($size->pathOnDisk($original->src, $original->path, $original->legacy_structure));
    }

    /**
     * Remove a size
     *
     * @param MediaUse $mediaUse
     * @param int $width
     * @param int $height
     * @return void
     */
    public function removeMediaUseSize(MediaUse $mediaUse, int $width, int $height): void
    {
        $size = new ImageSize($width, $height);
        Storage::disk(Config::get('filesystems.default_media'))
            ->delete($size->pathOnDisk(
                $mediaUse->seo_url ?? $mediaUse->media->src,
                $mediaUse->media->path,
                $mediaUse->media->legacy_structure
            ));
    }

    /**
     * Remove media file
     *
     * @param int|array $mediaId
     */
    public function destroyMedia(int|array $mediaId)
    {
        if (is_int($mediaId)) {
            $mediaId = [$mediaId];
        }

        $this->model->newQuery()
            ->with('uses')
            ->whereIn('id', $mediaId)
            ->get()
            ->each(function (Media $media) {
                if (!$media->external) {
                    $this->purgeFiles($media);
                }
                $media->uses()->delete();
                $media->delete();
            });
    }


    /**
     * Remove all files for allowed dimensions of this media file that are defined by this usage
     *
     * @param MediaUse $mediaUse
     * @return bool
     */
    public function removeMediaUseFiles(MediaUse $mediaUse): bool
    {
        $sizes_used_on_other_media_uses = $mediaUse
            ->media
            ->uses()->where('id', '!=', $mediaUse->id)->get()
            ->pluck('allowed_sizes')
            ->flatten()
            ->toArray();

        return collect($mediaUse->allowed_sizes)
            ->filter(fn($path) => $this->pathIsWidthAndHeight($path))
            ->filter(fn($path) => !in_array($path, $sizes_used_on_other_media_uses))
            ->map(fn($dimensions) => ImageSize::fromArray($this->pathStringToArray($dimensions)))
            ->reject(fn(ImageSize $size) => $this->removeFile($size->pathOnDisk(
                $mediaUse->media->src,
                $mediaUse->media->path,
                $mediaUse->media->legacy_structure
            )))
            ->isEmpty();
    }

    /**
     * Check if the path does represent width and height (or width-only with Auto)
     *
     * @param string $path
     * @return bool
     */
    public function pathIsWidthAndHeight(string $path): bool
    {
        return preg_match('/^([1-9][0-9]{0,3})x([1-9][0-9]{0,3}|Auto)$/', $path);
    }

    /**
     * Convert dimension string to an array
     *
     * @param string $dimensions
     * @return array
     */
    public function pathStringToArray(string $dimensions): array
    {
        list($width, $height) = explode('x', $dimensions);
        return [
            'width' => (int)$width,
            'height' => $height === 'Auto' ? null : (int)$height
        ];
    }

    /**
     * Get the prefix for storage path
     * (in case system needs to modify this - e.g. tenancy)
     *
     * @return string
     */
    public function storagePrefix(): string
    {
        return '';
    }

    /**
     * Generate a new size for the uploaded media file
     *
     * @param Media $media
     * @param ImageSize $size
     * @return mixed|void
     * @throws \Exception
     */
    public function createSize(Media $media, ImageSize $size, bool $overwrite = false)
    {
        $size_data = $this->getSize($media, $size);
        if ($overwrite === false && !empty($size_data)) {
            // Size exists and does not need to change - return image with existing data
            return $this->getImageManager('webp')
                ->make($size_data)
                ->response();
        }

        $original = $this->getOriginal($media);
        if (empty($original)) {
            // unable to crop as no original file exists
            return;
        }

        return $media->resizableMime()
            ? $this->makeSizeWithDimensions($media, $size, $original)
            : $this->copyToSizePath($media, $size);
    }

    public function changeMediaSize(MediaUse $use, ImageSize $from, ImageSize $to)
    {
        $this->removeMediaUseSize($use, $from->getWidth(), $from->getHeight());
        $this->createSize($use->media, $to);
    }

    /**
     * Check if asset file already exists in storage based on URL
     *
     * @param string $assetUrl
     * @return bool
     */
    public function assetExists(string $assetUrl): bool
    {
        $path = str_replace(
            substr(Storage::disk(Config::get('filesystems.default_media'))->url('/a'), 0, -2),
            '',
            $assetUrl
        );

        return Storage::disk(Config::get('filesystems.default_media'))->exists($path);
    }

    public function triggerSizeGenerationForMedia(Media $media, array $allowed_sizes, Model $model): void
    {
        if ($media->external) {
            return;
        }
        foreach ($allowed_sizes as $dimensions) {
            $size = ImageSize::fromArray($this->pathStringToArray($dimensions));
            if ($this->sizeExists($media, $size) === false) {
                $this->dispatch(new GenerateSizeForMediaFileJob($media, $size));
            }
        }
    }

    public function getAllowedSizes(Model $model, array $meta): array
    {
        if (!empty($model->data['options']['allowedSizes'])) {
            $model_dimensions = $model->data['options']['allowedSizes'];
        } else {
            $model_dimensions = MediaSize::query()
                ->where('model', $model->getMorphClass())
                ->whereNotNull('width')
                ->get()
                ->map(fn(MediaSize $size) => $size->dimensions)
                ->toArray();
        }

        return array_merge(
            $meta['allowed_sizes'] ?? [],
            config('media.thumbnail_sizes', []),
            $model_dimensions
        );
    }

    /**
     * Create Media record and size for an uploaded file
     *
     * @param string $uploadPath
     * @param string $fileName
     * @param string $mime
     * @param bool $image
     * @param ?int $creator
     * @return Media
     * @throws \Exception
     */
    protected function createMediaRecordForFile(
        string $uploadPath,
        string $fileName,
        string $mime,
        bool $image = false,
        ?int $creator = null,
        ?string $originalFilePath = null,
        ?string $provider = null,
    ): Media {
        /** @var Media $media */
        $media = $this->model->newQuery()
            ->updateOrCreate([
                'image_provider' => $provider,
                'src' => $fileName,
            ], [
                'path' => $uploadPath,
                'type' => $this->determineFileType($mime),
                'uploaded_by' => $creator,
                'source_filename' => $originalFilePath,
            ]);

        if ($image) {
            $this->createSize($media, ImageSize::fromArray(Config::get('media.default_thumbnail_size', [])));
        }
        return $media;
    }

    /**
     * Update Media record and size for an uploaded file
     *
     * @param Media $media
     * @param string $uploadPath
     * @param string $fileName
     * @param string $mime
     * @param bool $image
     * @param ?int $creator
     * @return Media
     * @throws \Exception
     */
    protected function updateRecordWithFile(
        Media $media,
        string $uploadPath,
        string $fileName,
        string $mime,
        bool $image = false,
        ?int $creator = null
    ): Media {
        $media->update([
            'path' => $uploadPath,
            'src' => $fileName,
            'type' => $this->determineFileType($mime),
            'uploaded_by' => $creator,
        ]);

        if ($image) {
            $this->createSize($media, ImageSize::fromArray(Config::get('media.default_thumbnail_size', [])), true);
            $this->dispatch(new RegenerateSizesForMediaFileJob($media));
        }
        return $media;
    }

    /**
     * Purge files before removing media record
     *
     * @param Media $media
     * @return void
     */
    protected function purgeFiles(Media $media): void
    {
        if ($media->type === 'image') {
            // Remove jpg copy
            $this->removeFile($media->path . '/' . str_replace('.webp', '.jpg', $media->src));
        }

        if (!$media->legacy_structure) {
            // Format is /tenant/Y-m/[orig|size]/ab/ab/filename
            $path_parts = explode('/orig/', $media->path);
            if (count($path_parts) == 2) {
                collect(Storage::disk('media')->directories($path_parts[0]))
                    ->each(fn($directory) => $this->removeFile($directory . '/' . $path_parts[1] . '/' . $media->src));
                return;
            }
        }

        $media->uses->each(function (MediaUse $mediaUse) {
            $this->removeMediaUseFiles($mediaUse);
            $mediaUse->delete();
        });

        collect($media->uses->pluck('dimension')->flatten(1))
            ->prepend($media->path . '/' . $media->src)
            ->prepend($media->thumbnailFilePath())
            ->each(fn($file) => $this->removeFile($file));
    }

    /**
     * Remove file from storage
     *
     * @param $path
     * @return bool
     */
    protected function removeFile($path): bool
    {
        if (empty($path)) {
            return true;
        }
        return Storage::disk(Config::get('filesystems.default_media'))->delete($path);
    }

    /**
     * Determine file type from mime
     *
     * @param string $mime
     * @return string
     */
    protected function determineFileType(string $mime): string
    {
        return collect(config('media.media_types'))
            ->filter(fn($section) => in_array($mime, $section))
            ->map(fn($mies, $name) => $name)
            ->first() ?? 'other';
    }

    /**
     * Check if size exists
     *
     * @param Media $media
     * @param ImageSize $size
     * @return bool
     * @throws \Exception
     */
    protected function sizeExists(Media $media, ImageSize $size): bool
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->exists($size->pathOnDisk($media->src, $media->path, $media->legacy_structure));
    }

    protected function getSize(Media $media, ImageSize $size): ?string
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->get($size->pathOnDisk($media->src, $media->path, $media->legacy_structure));
    }

    protected function getOriginal(Media $media): ?string
    {
        return Storage::disk(Config::get('filesystems.default_media'))->get($media->getOriginalFilePath());
    }

    protected function originalExists(Media $media): bool
    {
        return Storage::disk(Config::get('filesystems.default_media'))->exists($media->getOriginalFilePath());
    }

    /**
     * Create a size of an image based on dimensions
     *
     * @param Media $media
     * @param ImageSize $size
     * @return mixed|void
     * @throws \Exception
     */
    protected function makeSizeWithDimensions(Media $media, ImageSize $size, string $original)
    {
        $image = $this->getImageManager('webp')->make($original);

        if ($size->isWidthOnly()) {
            $image = $image->filter(new WidthResizeFilter($size->getWidth()));
        } else {
            $focus = $media->focal_point ? explode(',', $media->focal_point) : null;
            if (!empty($focus) && count($focus) == 2) {
                $image = $image->filter(new FocalCropFilter(
                    $size->getWidth(),
                    $size->getHeight(),
                    $focus[0],
                    $focus[1]
                ));
            } else {
                $image = $image->fit($size->getWidth(), $size->getHeight());
            }
        }

        Storage::disk(Config::get('filesystems.default_media'))
            ->put(
                $size->pathOnDisk($media->src, $media->path, $media->legacy_structure),
                $image->stream('webp'),
                ['visibility' => 'public']
            );

        return $image->response();
    }

    public function recropUsesToFocalPoint(Media $media): void
    {
        $this->dispatch(new ChangeSizesToMediaFocalPoint($media));
    }

    /**
     * Copy file to the relevant size path
     *
     * @param Media $media
     * @param ImageSize $size
     * @return bool
     */
    protected function copyToSizePath(Media $media, ImageSize $size)
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->copy($media->path . $media->src, $size->pathOnDisk($media->src, $media->path, $media->legacy_structure));
    }

    /**
     * Find requested file details - dimensions, type, path
     *
     * @param  $link_to_file
     * @return array|string[]
     */
    protected function getRequestedFileDetails($link_to_file): array
    {
        $file_details = pathinfo($link_to_file);
        $dimensions = pathinfo($file_details['dirname'], PATHINFO_BASENAME);

        // No extension means this is not a file access
        if (empty($file_details['extension'])) {
            throw new NotFoundHttpException('Resource is not a file', null, 404);
        }

        if ($this->pathIsWidthAndHeight($dimensions) == false) {
            return [
                'type' => 'original',
            ];
        }

        $dimensions = explode('x', $dimensions);
        return [
            'type' => 'size',
            'width' => $dimensions[0],
            'height' => $dimensions[1],
            'file_name' => $file_details['basename'],
        ];
    }

    /**
     * Check if dimensions are allowed for the media use
     *
     * @param MediaUse $mediaUse
     * @param $dimensions
     * @return bool
     */
    protected function areDimensionsAllowed(MediaUse $mediaUse, $dimensions): bool
    {
        return in_array($dimensions['width'] . 'x' . $dimensions['height'], $mediaUse->allowed_sizes);
    }


    protected function updateMediaMeta(Media $media, array $meta): void
    {
        $media->update([
            // only set if no meta already exists
            'alt_text' => $media->getAttribute('alt_text') ?? $meta['alt_text'] ?? null,
            'title' => $media->getAttribute('title') ?? $meta['title'] ?? null,
            'caption' => $media->getAttribute('caption') ?? $meta['caption'] ?? null,
            'description' => $media->getAttribute('description') ?? $meta['description'] ?? null,
        ]);
    }


    protected function setLastUsedOnMedia(array $mediaIds)
    {
        $this->model->newQuery()
            ->whereIn('id', $mediaIds)
            ->update([
                'last_used' => Carbon::now(),
            ]);
    }

    protected function setFolder(Model $model, array $mediaIds)
    {
        $folderName = match (get_class($model)) {
            Vehicle::class => 'Vehicles',
            VehicleOffer::class => 'Vehicle Offers',
            NewCar::class => 'New Vehicles',
            default => null,
        };

        if (!empty($folderName)) {
            $folder = $this->folder
                ->newQuery()
                ->firstOrCreate([
                    'name' => $folderName
                ]);

            $this->model->newQuery()
                ->whereIn('id', $mediaIds)
                ->update([
                    'folder_id' => $folder->id,
                ]);
        }
    }

    protected function setUsesForMedia(Media $media, $index, array $allowed_sizes, array $meta, Model $model): MediaUse
    {
        return $media->uses()
            ->updateOrCreate([
                'owner_type' => $model->getMorphClass(),
                'owner_id' => $model->id,
            ], [
                // update meta from passed with fallback of media object default
                'alt_text' => $meta['alt_text'] ?? $media->getAttribute('alt_text'),
                'title' => $meta['title'] ?? $media->getAttribute('title'),
                'caption' => $meta['caption'] ?? $media->getAttribute('caption'),
                'description' => $meta['description'] ?? $media->getAttribute('description'),
                'dimensions' => $meta['dimensions'] ?? null,
                'flags' => $meta['flags'] ?? null,
                'order' => $index,
                'primary' => $index == 0 && ($meta['primary'] ?? null),
                'secondary' => $meta['secondary'] ?? null,
                'allowed_sizes' => $allowed_sizes,
            ]);
    }


    public function createFolder(string $name, ?int $parent_id): Model
    {
        if (empty($parent_id)) {
            return $this->folder->newQuery()->create([
                'name' => $name
            ]);
        }
        /** @var MediaFolder $parent */
        $parent = $this->folder->newQuery()->findOrFail($parent_id);
        return $parent->children()->create([
            'name' => $name
        ]);
    }

    public function updateFolder(MediaFolder $folder, string $name, ?int $parent_id): MediaFolder
    {
        $folder->update(['name' => $name]);
        if ($folder->parent_id !== $parent_id) {
            if ($parent_id === null) {
                $folder->makeRoot();
            } else {
                $parent = $this->folder->newQuery()->findOrFail($parent_id);
                $folder->parent()->associate($parent)->save();
            }
        }
        return $folder;
    }

    public function deleteFolder(MediaFolder $folder): bool
    {
        $folder->media()->update([
            'folder_id' => $folder->parent_id,
        ]);
        return $folder->delete();
    }

    /**
     * Get list of folders in library
     *
     * @param Request $request
     *
     */
    public function getFolders(Request $request)
    {
        return $this->folder
            ->orderBy('order')
            ->get()
            ->toTree();
    }

    /**
     * Reorder folders
     *
     * @param $folders
     * @return void
     */
    public function reorderFolders(array $folders): void
    {
        app(MediaFolder::class)::rebuildTree($folders);

        $traverse = function ($folders) use (&$traverse) {
            foreach ($folders as $index => $folder) {
                app(MediaFolder::class)->where('id', $folder['id'])->update([
                    'order' => $index
                ]);

                if (!empty($folder['children'])) {
                    $traverse($folder['children']);
                }
            }
        };
        $traverse($folders);
    }

    /**
     * Add Images To Folder
     *
     * @return void
     */
    public function addImagesToFolder(int $folder, array $images): void
    {
        $this->model->newQuery()
            ->whereIn('id', $images)
            ->update([
                'folder_id' => $folder
            ]);
    }

    public function uploadZip(MediaUploadRequest $request): Media
    {
        $uniqueFolder = uniqid('zip_extract_', true);
        $extractPath = Storage::disk(config('filesystems.default_media'))->path($uniqueFolder);

        $zipPath = $this->extractZipArchive($request->file('file'), $extractPath);
        $files = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($extractPath));

        $folderMap = $this->mapArchiveFolders(
            $this->getArchiveDirectories($files, $extractPath),
            $request->input('folder')
        );

        $mediaFolderAssignment = $this->uploadArchiveImages($files, $request, $extractPath, $folderMap);
        foreach ($mediaFolderAssignment as $folderId => $images) {
            $this->addImagesToFolder($folderId, $images);
        }

        Storage::disk(config('filesystems.default_media'))->deleteDirectory($uniqueFolder);
        return $this->model->newInstance([
            'type' => 'zip',
            'src' => $request->file('file')->getClientOriginalName(),
            'path' => $zipPath,
            'folder_id' => $request->input('folder')
        ]);
    }

    private function extractZipArchive(UploadedFile $file, string $extractPath): string
    {
        $zip = new \ZipArchive();
        $zipPath = $file->getPathname();

        if ($zip->open($zipPath) === true) {
            $zip->extractTo($extractPath);
            $zip->close();
        } else {
            throw new \Exception("Unable to extract the ZIP file");
        }

        return $zipPath;
    }

    private function getArchiveDirectories(\RecursiveIteratorIterator $files, string $extractPath): array
    {
        $directories = [];

        foreach ($files as $key => $file) {
            if ($file->isDir()) {
                $relativePath = str_replace($extractPath, '', $file->getPath());
                $folderPath = trim($relativePath, '/');
                $folderName = basename($folderPath);

                if (!empty($folderName) && !str_starts_with($folderName, '._') && $folderName !== '__MACOSX') {
                    $directories[] = $folderPath;
                }
            }
        }

        usort($directories, function ($a, $b) {
            return substr_count($a, '/') - substr_count($b, '/');
        });

        return $directories;
    }

    private function mapArchiveFolders(array $directories, ?int $parentFolderId): array
    {
        $folderMap = [];

        foreach ($directories as $folderPath) {
            $folderName = basename($folderPath);
            $parentPath = dirname($folderPath);
            $parentFolderId = $parentPath ? ($folderMap[$parentPath] ?? $parentFolderId) : null;

            $mediaFolder = $this->folder->newQuery()
                ->where('parent_id', $parentFolderId)
                ->where('name', $folderName)
                ->first();

            if (!$mediaFolder) {
                $mediaFolder = $this->createFolder($folderName, $parentFolderId);
            }

            $folderMap[$folderPath] = $mediaFolder->id;
        }

        return $folderMap;
    }

    private function uploadArchiveImages(
        \RecursiveIteratorIterator $files,
        MediaUploadRequest $request,
        string $extractPath,
        array $folderMap
    ): array {
        $mediaStructure = [];

        foreach ($files as $file) {
            if (!$file->isFile()) {
                continue;
            }

            $fileName = basename($file->getFilename());
            if (str_starts_with($fileName, '._') || str_starts_with($fileName, '.')) {
                continue;
            }

            $media = $this->uploadImage($request, null, $file->getPathname());
            $filePath = trim(str_replace($extractPath, '', $file->getPath()), '/');

            if (array_key_exists($filePath, $folderMap)) {
                $folderId = $folderMap[$filePath];
                $mediaStructure[$folderId][] = $media->id;
            }
        }

        return $mediaStructure;
    }

    private function getImageManager(string $image_format = 'jpg'): ImageManager
    {
        if (in_array($image_format, self::$backup_image_formats)) {
            return (new ImageManager(Config::get('media.backup_image_manager_config', [])));
        }
        return (new ImageManager(Config::get('media.image_manager_config', [])));
    }
}
