<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\ContentManager\Facades\Page;
use Mtc\ContentManager\Http\Requests\BulkUpdatePagesRequest;
use Mtc\ContentManager\Http\Requests\CheckPageSlugRequest;
use Mtc\ContentManager\Http\Requests\CopyPageRequest;
use Mtc\ContentManager\Http\Requests\StorePageRequest;
use Mtc\ContentManager\Http\Requests\UpdatePageRequest;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\Http\Requests\UpdateVersionRequest;
use Mtc\ContentManager\Http\Resources\PageViewResource;

class PageController
{
    use ValidatesRequests;

    /**
     * Show list of pages
     *
     * @param Request $request
     * @param PageModel $pageModel
     * @param Template $template
     * @return array
     */
    public function index(Request $request, PageModel $pageModel, Template $template): JsonResource
    {
        $perPage = min((int) $request->input('per_page', 15), 200);

        $pages = $pageModel->newQuery()
            ->with([
                'primaryMediaUse.media',
                'users',
                'template',
            ])
            ->when($request->input('my-content'), function ($query) {
                $query->whereHas('users', function ($user_query) {
                    $user_query->where('users.id', Auth::id());
                });
            })
            ->withCommentCount()
            ->setSortBy($request->input('sort_by'))
            ->setFilters($request->input('filters') ?? '')
            ->paginate($perPage);

        $pageResource = Config::get('pages.page_list_resource');
        return new $pageResource($pages);
    }

    public function indexVersions(Request $request, VersionModel $versionModel)
    {
        $users = [];
        $resource = Config::get('pages.version_list_resource');
        $term = $request->input('search_term');
        if (!empty($term)) {
            $userClass = Config::get('auth.providers.users.model');
            $users = tenant()->users()
                ->where('name', 'like', "%$term%")
                ->pluck('user_id')
                ->merge($userClass::query()->where('name', 'like', "%$term%")->role('mtc')->pluck('id'));
        }
        return new $resource(
            $versionModel->newQuery()
                ->setSortBy($request->input('sort_by'))
                ->when(
                    !empty($term),
                    fn($query) => $query
                        ->where(fn($termQuery) => $termQuery
                            ->whereHas('page', fn($pageQuery) => $pageQuery
                                ->where('slug', 'like', "$term%")
                                ->orWhere('title', 'like', "%$term%"))
                            ->orWhereIn('author_id', $users)
                            ->orWhere('id', 'like', "%$term%"))
                )
                ->with([
                    'page.primaryMediaUse.media',
                    'author'
                ])
                ->paginate()
        );
    }

    /**
     * Store a new page
     *
     * @param StorePageRequest $request
     * @return Model
     */
    public function store(StorePageRequest $request): Model
    {
        return Page::create($request->input('title'), $request->input('template_id'), $request->input('category'));
    }

    /**
     * Show a page record
     *
     * @param PageModel $pageModel
     * @param int $page
     * @return array
     */
    public function show(PageModel $pageModel, int $pageId): PageViewResource
    {
        $page = $pageModel
            ->withTrashed()
            ->find($pageId);

        $pageResource = Config::get('pages.page_view_resource');
        return new $pageResource($page);
    }

    /**
     * Show a page version record
     *
     * @param PageModel $pageModel
     * @param int $page
     * @return array
     */
    public function showVersion(VersionModel $versionModel, int $pageId, int $versionId)
    {
        $version = $versionModel->newQuery()->find($versionId);
        $versionResource = Config::get('pages.version_view_resource');
        return new $versionResource($version);
    }

    /**
     * Update a page
     *
     * @param UpdatePageRequest $request
     * @param PageModel $page
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    public function update(UpdatePageRequest $request, PageModel $page)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        Page::save($page, $request->input(), Auth::id());

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $page->refresh();

        $pageResource = Config::get('pages.page_view_resource');
        return new $pageResource($page);
    }

    /**
     * Save a version of page
     *
     * @param UpdateVersionRequest $request
     * @param PageModel $page
     * @return Response
     */
    public function createVersion(UpdateVersionRequest $request, PageModel $page)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        try {
            $version = Page::savePageVersion($page, $request->input('content'), Auth::id());
        } catch (\Exception $exception) {
            Log::error('Issue creating page version ', [
                'page_id' => $page->id,
                'error' => $exception->getMessage(),
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $versionResource = Config::get('pages.version_view_resource');
        return new $versionResource($version);
    }

    /**
     * Save a version of page
     *
     * @param UpdateVersionRequest $request
     * @param PageModel $page
     * @param VersionModel $version
     * @return Response
     */
    public function saveVersion(UpdateVersionRequest $request, PageModel $page, VersionModel $version)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        try {
            Page::saveVersion($version, $request->input(), Auth::id());
        } catch (\Exception $exception) {
            Log::error('Issue saving page version ', [
                'page_id' => $page->id,
                'version_id' => $version->id,
                'error' => $exception->getMessage(),''
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $page->refresh();

        $pageResource = Config::get('pages.page_view_resource');
        return new $pageResource($page);
    }

    /**
     * Mark page as deleted
     *
     * @param int $pageId
     * @return Response
     */
    public function destroy(int $pageId)
    {
        if (Page::canRemove($pageId)) {
            return response([
                'success' => Page::remove($pageId) > 0,
            ]);
        }

        return response([
            'message' => __('validation.not_allowed_to_delete')
        ], 422);
    }

    /**
     * Remove pages from trash
     *
     * @param BulkUpdatePagesRequest $request
     * @return array
     */
    public function emptyTrash(BulkUpdatePagesRequest $request)
    {
        return [
            'success' => Page::remove($request->input('id', []), true) > 0,
        ];
    }

    /**
     * Check if slug is available for page
     *
     * @param CheckPageSlugRequest $request
     * @return string
     */
    public function checkSlug(CheckPageSlugRequest $request): string
    {
        // Request validation performs check
        return 'ok';
    }

    /**
     * Restore page from trash
     *
     * @param int $pageId
     * @return Response
     */
    public function restore(int $pageId): Response
    {
        try {
            Page::restorePage($pageId);
        } catch (ModelNotFoundException $exception) {
            return response([
                'success' => false,
                'page_id' => $pageId,
                'message' => $exception->getMessage(),
            ], 404);
        } catch (\Exception $exception) {
            return response([
                'success' => false,
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
                'message' => $exception->getMessage(),
            ], 500);
        }

        return response([
            'success' => true,
        ]);
    }

    public function syncWithTemplate(PageModel $pageModel, int $pageId): Response|PageViewResource
    {
        try {
            $version = Page::syncContentWithTemplate($pageId);
            if (!empty($version)) {
                return response([
                    'success' => true,
                    'version' => $version->id,
                ], 201);
            }
        } catch (\Exception $exception) {
            Log::error('Failed to sync page structure ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        $page = $pageModel->find($pageId);

        $pageResource = Config::get('pages.page_view_resource');
        return new $pageResource($page);
    }

    /**
     * Restore page from trash
     *
     * @param int $pageId
     * @return Response
     */
    public function restoreVersion(int $pageId, int $versionId): Response|JsonResource
    {
        try {
            $version = Page::restoreVersion($versionId);
        } catch (\Exception $exception) {
            Log::error('Failed to restore page version ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        $versionResource = Config::get('pages.version_view_resource');
        return new $versionResource($version);
    }

    public function markVersionForReview(int $pageId, int $versionId): Response|JsonResource
    {
        try {
            $version = Page::markVersionForReview($pageId, $versionId);
        } catch (\Exception $exception) {
            Log::error('Failed to mark page version for review ', [
                'page_id' => $pageId,
                'version_id' => $versionId,
                'error' => $exception->getMessage(),
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        $versionResource = Config::get('pages.version_view_resource');
        return new $versionResource($version);
    }

    public function requestChanges(int $pageId, int $versionId): Response|JsonResource
    {
        try {
            $version = Page::markVersionAsChangesRequested($pageId, $versionId);
        } catch (\Exception $exception) {
            Log::error('Failed to mark page version changes requested ', [
                'page_id' => $pageId,
                'version_id' => $versionId,
                'error' => $exception->getMessage(),
            ]);
            return \response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        $versionResource = Config::get('pages.version_view_resource');
        return new $versionResource($version);
    }

    /**
     * Create a copy of the page
     *
     * @param CopyPageRequest $request
     * @param int $pageId
     * @return Response
     */
    public function copy(CopyPageRequest $request, int $pageId): Response|JsonResource
    {
        try {
            $page = Page::copyPage($pageId, $request->input('title'), $request->input('with_content'));
        } catch (\Exception $exception) {
            Log::error('Failed to copy a page ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return response([
                'success' => false,
                'message' => $exception->getMessage(),
            ], 400);
        }

        $pageResource = Config::get('pages.page_view_resource');
        return new $pageResource($page);
    }
}
