<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\CommentRepository;
use Mtc\ContentManager\Contracts\Comment;
use Mtc\ContentManager\Http\Requests\StoreCommentRequest;

class CommentController
{
    use ValidatesRequests;

    public function show(CommentRepository $repository, string $commentable, int $id)
    {
        $commentResource = Config::get('pages.comment_list_resource');
        return new $commentResource($repository->get($commentable, $id));
    }
    /**
     * Store a comment
     *
     * @param StoreCommentRequest $request
     * @param CommentRepository $repository
     * @return JsonResource
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(StoreCommentRequest $request, CommentRepository $repository): JsonResource
    {
        $model = $this->findLinkedModelInstance($request);
        if ($model) {
            $repository->add($request->input('text'), Auth::id(), $model, $request->input('parent_id'));
        }
        $commentResource = Config::get('pages.comment_list_resource');

        return new $commentResource($repository->getForModel($model));
    }

    /**
     * Mark comments as resolved
     *
     * @param Comment $comment
     * @param CommentRepository $repository
     * @return JsonResource
     */
    public function markResolved(Comment $comment, CommentRepository $repository): JsonResource
    {
        $commentResource = Config::get('pages.comment_list_resource');
        $repository->markResolved($comment->id);
        return new $commentResource($repository->getForModel($comment->commentable));
    }

    /**
     * Fin the object that is commented
     *
     * @param Request $request
     * @return null|Model
     */
    protected function findLinkedModelInstance(Request $request): ?Model
    {
        return match ($request->input('commentable_type')) {
            'page' => App::make(Config::get('pages.page_model'))
                ->newQuery()
                ->findOrFail($request->input('commentable_id')),
            'version' => App::make(Config::get('pages.version_model'))
                ->newQuery()
                ->findOrFail($request->input('commentable_id')),
            'template' => App::make(Config::get('pages.template_model'))
                ->newQuery()
                ->findOrFail($request->input('commentable_id')),
            default => null,
        };
    }
}
