<?php

namespace Mtc\ContentManager\Contracts;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Kalnoy\Nestedset\NodeTrait;
use Kalnoy\Nestedset\QueryBuilder;
use Mtc\ContentManager\Casts\SeoDataCast;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\PageFactory;
use Mtc\ContentManager\Models\Comment;
use Mtc\ContentManager\Models\ContentCategory;
use Mtc\ContentManager\Models\ContentTag;
use Mtc\ContentManager\Models\PageUser;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\ContentManager\Traits\ModelSortAndFilter;

/**
 * @param Content $content
 */
abstract class PageModel extends Model implements ModelWithContent
{
    use SoftDeletes;
    use HasFactory;
    use ModelSortAndFilter;
    use HasMedia;
    use NodeTrait;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'pages';

    /**
     * @var string[]
     */
    protected $guarded = [
        'id'
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'seo' => SeoDataCast::class,
        'text' => 'array',
        'meta' => 'array',
        'data' => 'array',
        'published_at' => 'datetime',
        'unpublished_at' => 'datetime',
    ];

    protected $appends = [
        'updated_at_formatted'
    ];

    /**
     * Define model factory
     *
     * @return PageFactory
     */
    protected static function newFactory()
    {
        return PageFactory::new();
    }

    /**
     * Relationship with page versions
     *
     * @return HasMany
     */
    public function versions(): HasMany
    {
        return $this->hasMany(Config::get('pages.version_model'));
    }

    public function activeVersion(): HasOne
    {
        return $this->hasOne(Config::get('pages.version_model'))
            ->where('is_active', true);
    }

    /**
     * Relationship with template that defines structure of page
     *
     * @return BelongsTo
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_model'));
    }

    /**
     * Relationship with comments that have been set on the page
     *
     * @return MorphMany
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Config::get('pages.comment_model'), 'commentable');
    }

    /**
     * Relationship with content elements on page
     *
     * @return HasMany
     */
    public function content(): HasMany
    {
        return $this->hasMany(Config::get('pages.content_model'), 'page_id')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content elements on page
     *
     * @return HasMany
     */
    public function allContent(): HasMany
    {
        return $this->hasMany(Config::get('pages.content_model'), 'page_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content elements on page
     *
     * @return HasMany
     */
    public function rootLevelContent(): HasMany
    {
        return $this->content()->whereNull('parent_id');
    }

    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(
            ContentTag::class,
            'page_tags',
            'page_id',
            'tag_id'
        );
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(
            ContentCategory::class,
            'page_categories',
            'page_id',
            'category_id'
        );
    }

    /**
     * Relationship with users that are involved in a page
     *
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        $pivot = new PageUser();
        if (app()->runningUnitTests()) {
            $pivot_table = $pivot->getTable();
        } else {
            $pivot_table = $pivot->getConnection()->getDatabaseName() . '.' . $pivot->getTable();
        }
        return $this->belongsToMany(Config::get('auth.providers.users.model'), $pivot_table);
    }

    /**
     * Add comment count to data query
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithCommentCount(Builder $query): Builder
    {
        return $query->addSelect([
            'comment_count' => Comment::query()
                ->join('page_content', 'page_content.id', '=', 'commentable_id')
                ->whereColumn('commentable_id', $this->getTable() . '.id')
                ->where('commentable_type', $this->getMorphClass())
                ->selectRaw('COUNT(*)')
        ]);
    }

    public function scopeSearchByTerm(QueryBuilder $query, $term)
    {
        return $query->where(fn($termQuery) => $termQuery->where('slug', 'like', "$term%")
            ->orWhere('title', 'like', "%$term%")
            ->orWhereFullText('search_content', $term));
    }

    /**
     * Scope for getting current users pages
     *
     * @param Builder $query
     * @return void
     */
    public function scopeMine(Builder $query)
    {
        $query->whereHas('users', fn ($query) => $query->where('user_id', auth()->id()));
    }

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', 'published')
            ->where(fn($query) => $query
                ->whereNull('unpublished_at')
                ->orWhere('unpublished_at', '>', Carbon::now()));
    }

    /**
     * Apply searched name to page query
     *
     * @param Builder $query
     * @param string|null $search_name
     * @return Builder
     */
    public function scopeSearchName(Builder $query, string|null $search_name): Builder
    {
        if (empty($search_name)) {
            return $query;
        }

        return $query->where(function (Builder $query) use ($search_name) {
            $query->where('title', 'like', "%$search_name%");
        });
    }

    /**
     * add deleted() scope to query
     *
     * @param Builder $builder
     * @return Builder
     */
    public function scopeDeleted(Builder $builder): Builder
    {
        return $builder->onlyTrashed();
    }

    /**
     * updated_at_formatted for easy display
     *
     * @return mixed
     */
    public function getUpdatedAtFormattedAttribute()
    {
        return $this->updated_at?->diffForHumans();
    }

    /**
     * Define thumbnail sizes to auto-generate for this model
     *
     * @return mixed
     */
    public function getDefaultAllowedMediaSizesAttribute()
    {
        return Config::get('pages.thumbnail_sizes', []);
    }

    public function getOwnerColumn(): string
    {
        return 'page_id';
    }
}
