<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Mtc\ContentManager\Facades\Page;
use Mtc\ContentManager\Http\Requests\BulkUpdatePagesRequest;
use Mtc\ContentManager\Http\Requests\CheckPageSlugRequest;
use Mtc\ContentManager\Http\Requests\CopyPageRequest;
use Mtc\ContentManager\Http\Requests\StorePageRequest;
use Mtc\ContentManager\Http\Requests\UpdatePageRequest;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\PageRepository;

class PageController
{
    use ValidatesRequests;

    /**
     * Show list of pages
     *
     * @param Request $request
     * @param PageModel $pageModel
     * @param Template $template
     * @return array
     */
    public function index(Request $request, PageModel $pageModel, Template $template): array
    {
        $pages = $pageModel->newQuery()
            ->with([
                'users',
                'template',
            ])
            ->when($request->input('my-content'), function ($query) {
                $query->whereHas('users', function ($user_query) {
                    $user_query->where('users.id', Auth::id());
                });
            })
            ->deleted($request->has('trashed') && $request->input('trashed'))
            ->withCommentCount()
            ->latest()
            ->get();

        return [
            'pages' => $pages,
            'templates' => $template->newQuery()->get(),
        ];
    }

    /**
     * Store a new page
     *
     * @param StorePageRequest $request
     * @param PageRepository $repository
     * @return Model
     */
    public function store(StorePageRequest $request, PageRepository $repository): Model
    {
        return $repository->create($request->input('title'), $request->input('template_id'));
    }

    /**
     * Show a page record
     *
     * @param PageModel $pageModel
     * @param int $page
     * @return array
     */
    public function show(PageModel $pageModel, int $pageId): array
    {
        $page = $pageModel
            ->withTrashed()
            ->with([
                'content.mediaUses.media',
                'content.subContent.mediaUse.media',
                'content.subContent.subContent.mediaUse.media',
                'content.globalContent.contentElement.fields',
                'content.contentElement.fields',
                'content.subContent.contentElement.fields',
                'content.subContent.subContent.contentElement.fields',
            ])
            ->find($pageId);

        return [
            'page' => $page,
            'page_versions' => $page->versions()->latest()->take(5)->get()
        ];
    }

    /**
     * Update a page
     *
     * @param UpdatePageRequest $request
     * @param PageModel $page
     * @param PageRepository $repository
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    public function update(UpdatePageRequest $request, PageModel $page, PageRepository $repository)
    {
        DB::beginTransaction();
        $repository->save($page, $request->input(), Auth::id());
        DB::commit();

        $page->refresh();
        return [
            'success' => true,
            'page' => $page,
        ];
    }

    /**
     * Mark page as deleted
     *
     * @param int $pageId
     * @return array
     */
    public function destroy(int $pageId)
    {
        return [
            'success' => Page::remove($pageId) > 0,
        ];
    }

    /**
     * Remove pages from trash
     *
     * @param BulkUpdatePagesRequest $request
     * @param PageRepository $repository
     * @return array
     */
    public function emptyTrash(BulkUpdatePagesRequest $request, PageRepository $repository)
    {
        return [
            'success' => $repository->remove($request->input('id', []), true) > 0,
        ];
    }

    /**
     * Check if slug is available for page
     *
     * @param CheckPageSlugRequest $request
     * @return string
     */
    public function checkSlug(CheckPageSlugRequest $request): string
    {
        // Request validation performs check
        return 'ok';
    }

    /**
     * Restore page from trash
     *
     * @param PageRepository $repository
     * @param int $pageId
     * @return array
     */
    public function restore(PageRepository $repository, int $pageId): array
    {
        try {
            $repository->restorePage($pageId);
        } catch (\Exception $exception) {
            Log::error('Failed to restore a deleted page ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return [
                'success' => false,
                'error' => $exception->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }

    /**
     * Restore page from trash
     *
     * @param PageRepository $repository
     * @param int $pageId
     * @return array
     */
    public function restoreVersion(PageRepository $repository, int $pageId, int $versionId): array
    {
        try {
            $repository->restoreVersion($versionId);
        } catch (\Exception $exception) {
            Log::error('Failed to restore page version ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return [
                'success' => false,
                'error' => $exception->getMessage(),
            ];
        }

        return [
            'success' => true,
        ];
    }

    /**
     * Create a copy of the page
     *
     * @param CopyPageRequest $request
     * @param PageRepository $repository
     * @param int $pageId
     * @return array
     */
    public function copy(CopyPageRequest $request, PageRepository $repository, int $pageId)
    {
        try {
            $page = $repository->copyPage($pageId, $request->input('title'), $request->input('with_content'));
        } catch (\Exception $exception) {
            Log::error('Failed to copy a page ', [
                'page_id' => $pageId,
                'error' => $exception->getMessage(),
            ]);
            return [
                'success' => false,
                'error' => $exception->getMessage(),
            ];
        }

        return [
            'success' => true,
            'page' => $page,
        ];
    }
}
