<?php

namespace Mtc\ContentManager;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use JetBrains\PhpStorm\ArrayShape;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Menu;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;

class ElementRepository
{
    public function __construct(
        protected ContentElement $elementModel
    ) {
        //
    }

    /**
     * Attach model to repository
     *
     * @param ContentElement $element
     * @return $this
     */
    public function setModel(ContentElement $element): self
    {
        $this->elementModel = $element;
        return $this;
    }

    /**
     * Find the element
     *
     * @param int $id
     * @return ContentElement|Model
     */
    public function find(int $id): ContentElement
    {
        return $this->elementModel->newQuery()->findOrFail($id);
    }

    /**
     * Load element by id for usage
     *
     * @param int $id
     * @return $this
     */
    public function load(int $id): self
    {
        $this->elementModel = $this->elementModel->newQuery()->findOrFail($id);
        return $this;
    }

    /**
     * Get list
     *
     * @param string|null $searchTerm
     * @return LengthAwarePaginator
     */
    public function getList(string $searchTerm = null): LengthAwarePaginator
    {
        return $this->elementModel->newQuery()
            ->when(!empty($searchTerm), fn($query) => $query->searchTerm($searchTerm))
            ->latest()
            ->paginate();
    }

    /**
     * Create a new content element
     *
     * @param string $title
     * @param string $subtitle
     * @param string $icon
     * @param string $category
     * @param bool $enabled
     * @return ContentElement|Model
     */
    public function create(
        string $title,
        string $subtitle = null,
        string $icon = null,
        string $category = null,
        bool $enabled = true,
    ): ContentElement {
        $this->elementModel = $this->elementModel->newQuery()
            ->create([
                'title' => $title,
                'subtitle' => $subtitle,
                'icon' => $icon,
                'category' => $category,
                'is_enabled' => $enabled,
            ]);

        return $this->elementModel;
    }

    public function update(
        ContentElement $element,
        string $title,
        string $subtitle = null,
        string $icon = null,
        string $category = null,
        bool $enabled = true,
        array $fields = [],
    ): ContentElement {
        $this->setModel($element);
        $element->fill([
            'title' => $title,
            'subtitle' => $subtitle,
            'icon' => $icon,
            'category' => $category,
            'is_enabled' => $enabled,
        ])->save();

        $this->syncFields(collect($fields));
        return $element;
    }

    public function makeCopy(int $elementId, string $title): ContentElement
    {
        $original = $this->find($elementId);

        $this->elementModel = $this->create(
            title: $title,
            subtitle: $original->subtitle,
            icon: $original->icon,
            category: $original->category,
            enabled: false
        );

        $original->fields
            ->each(fn (ContentElementField $field) => $this->addField([
                'child_element_id' => $field->child_element_id,
                'type' => $field->field_type,
                'name' => $field->name,
                'data' => $field->data,
                'choices' => $field->choices,
            ]));

        return $this->elementModel;
    }

    /**
     * Add field to content element
     *
     * @param array $field
     * @return $this
     */
    #[ArrayShape([
        'child_element_id' => "int",
        'type' => "string",
        'name' => "string",
        'data' => "array",
        'choices' => "array",
    ])]
    public function addField(array $field): self
    {
        $this->elementModel->fields()
            ->create([
                'child_element_id' => $field['child_element_id'] ?? null,
                'field_type' => !empty($field['child_element_id']) ? 'element' : $field['type'],
                'name' => $field['name'],
                'data' => json_encode($field['data'] ?? []),
                'choices' => json_encode($field['choices'] ?? []),
            ]);

        return $this;
    }

    /**
     * Remove a content element
     *
     * @param int $elementId
     * @return bool
     */
    public function destroy(int $elementId): bool
    {
        return $this->elementModel
            ->newQuery()
            ->where('id', $elementId)
            ->delete();
    }

    /**
     * Get the list of places where content element is used
     *
     * @param int $elementId
     * @return array
     */
    public function getUsage(int $elementId): array
    {
        return [
            'templates' => $this->findTemplatesUsingElement($elementId),
            'freeform_pages' => $this->findPagesUsingElement($elementId),
            'global_elements' => $this->findGlobalElementsUsingElement($elementId),
            'menus' => $this->findMenusUsingElement($elementId),
        ];
    }

    /**
     * Show full usage list of a specific content type
     *
     * @param int $elementId
     * @param string $type
     * @return Collection
     */
    public function fullUsageList(int $elementId, string $type): Collection
    {
        return match ($type) {
            'templates' => $this->findTemplatesUsingElement($elementId, 0),
            'freeform_pages' => $this->findPagesUsingElement($elementId, 0),
            'global_elements' => $this->findGlobalElementsUsingElement($elementId, 0),
            'menus' => $this->findMenusUsingElement($elementId, 0),
        };
    }

    /**
     * Find Template records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findTemplatesUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Template::query()
            ->whereHas('elements', fn($query) => $query->where('element_id', $elementId))
            ->withActivePageCount()
            ->withInactivePageCount()
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Find free-form pages (no template) records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findPagesUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Page::query()
            ->whereNull('template_id')
            ->whereHas('content', fn($query) => $query->where('element_id', $elementId))
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findGlobalElementsUsingElement(int $elementId, int $limit = 10): Collection
    {
        return GlobalContent::query()
            ->where('element_id', $elementId)
            ->withTemplateCount()
            ->withFreeformPageCount()
            ->limit($limit ?: null)
            ->get();
    }


    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findMenusUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Menu::query()
            ->whereHas('entries', fn ($query) => $query->where('element_id', $elementId))
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Sync fields of a content element
     *
     * @param Collection $fields
     * @return void
     */
    protected function syncFields(Collection $fields)
    {
        $requestEntryIds = $fields->pluck('id');
        $currentFields = $this->elementModel->fields()->get();

        $currentFields->reject(fn($field) => in_array($field->id, $requestEntryIds))
            ->each(fn($field) => $field->delete());

        $currentFields->filter(fn($field) => in_array($field->id, $requestEntryIds))
            ->each(fn($field) => $field->fill($fields->where('id', $field->id)->first())->save());

        $fields->reject(fn($field) => !empty($field['id']))
            ->each(fn($field) => $this->addField($field));
    }
}
