<?php

namespace Mtc\ContentManager;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\Menu;
use Mtc\ContentManager\Contracts\MenuEntry;

class MenuRepository
{
    public function update(Menu $menu, Request $request)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $menu->fill($request->validated())->save();
        $this->syncTopLevelEntries($menu, collect($request->input('entries', [])));

        if (config('pages.use_transactions')) {
            DB::commit();
        }
        return $menu;
    }

    /**
     * Sync all menu entries of this level
     *
     * @param Model $menu
     * @param Collection $entries
     * @return void
     */
    protected function syncTopLevelEntries(Menu $menu, Collection $entries): void
    {
        $requestEntryIds = $entries->pluck('id');
        $currentEntries = $menu->entries()->get();

        // Drop
        $currentEntries->reject(fn($entry) => in_array($entry->id, $requestEntryIds))
            ->each(fn($entry) => $entry->delete());

        // Update
        $currentEntries->filter(fn($entry) => in_array($entry->id, $requestEntryIds))
            ->each(function ($entry) use ($entries, $menu) {
                $matched = $entries->where('id', $entry->id)->first();
                $matched['menu_id'] = $menu->id;
                $matched['parent_id'] = null;
                $entry->fill($matched)->save();
                if (!empty($matched['children'])) {
                    $this->syncRecursiveEntries($menu, $entry, collect($matched['children']));
                }
            });

        // Add
        $entries->reject(fn($entry) => !empty($entry['id']))
            ->each(function ($entry_data) use ($menu) {
                $entry_data['menu_id'] = $menu->id;
                $new_entry = $menu->entries()->create($entry_data);
                if (!empty($entry_data['children'])) {
                    $this->syncRecursiveEntries($menu, $new_entry, collect($entry_data['children']));
                }
            });
    }

    protected function syncRecursiveEntries(Menu $menu, MenuEntry $parent, Collection $entries)
    {
        $requestEntryIds = $entries->pluck('id');
        $currentEntries = $parent->children()->get();

        // Drop
        $currentEntries->reject(fn($entry) => in_array($entry->id, $requestEntryIds))
            ->each(fn($entry) => $entry->delete());

        // Update
        $currentEntries->filter(fn($entry) => in_array($entry->id, $requestEntryIds))
            ->each(function ($entry) use ($entries, $menu, $parent) {
                $matched = $entries->where('id', $entry->id)->first();
                $matched['menu_id'] = $menu->id;
                $matched['parent_id'] = $parent->id;
                $entry->fill($matched)->save();
                if (!empty($matched['children'])) {
                    $this->syncRecursiveEntries($menu, $entry, collect($matched['children']));
                }
            });

        // Add
        $entries->reject(fn($entry) => !empty($entry['id']))
            ->each(function ($entry_data) use ($menu, $parent) {
                $entry_data['menu_id'] = $menu->id;
                /** @var MenuEntry $new_entry */
                $new_entry = $parent->children()->create($entry_data);
                if (!empty($entry_data['children'])) {
                    $this->syncRecursiveEntries($menu, $new_entry, collect($entry_data['children']));
                }
            });
    }
}
