<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class GlobalElementControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();

        GlobalContent::factory(5)->create();
        $response = $this->getJson(route('content.global-elements.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, count($response->json('data')));
    }

    public function testStore()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        $response = $this->postJson(route('content.global-elements.store'), [
            'name' => 'lorem ipsum',
            'element_id' => $element->id,
        ]);

        $response->assertStatus(201);
        $this->assertTrue(GlobalContent::query()->where('name', 'lorem ipsum')->exists());
    }

    public function testShow()
    {
        $this->asUser();

        $content = GlobalContent::factory()->create();
        $response = $this->getJson(route('content.global-elements.show', $content));

        $response->assertStatus(200);
        $this->assertEquals($content->id, $response->json('data.id'));
        $this->assertArrayHasKey('content', $response->json('data'));
    }

    public function testUpdate()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'baz',
        ]);
        $content = GlobalContent::factory()->create(['element_id' => $element->id]);
        $response = $this->putJson(route('content.global-elements.update', $content), [
            'name' => 'foo',
            'element_id' => $element->id,
            'content' => [
                [
                    'id' => 'abc123',
                    'slug' => 'foo',
                    'name' => 'foo',
                    'content' => 'something',
                ],
                [
                    'id' => 'dev231',
                    'slug' => 'baz',
                    'name' => 'baz',
                    'content' => 'lorem ipsum',
                ],
            ],
        ]);

        $response->assertStatus(200);
    }

    public function testDestroy()
    {
        $this->asUser();

        $content = GlobalContent::factory()->create();
        $response = $this->deleteJson(route('content.global-elements.destroy', $content));

        $response->assertStatus(200);
        $this->assertFalse(GlobalContent::query()->where('id', $content->id)->exists());
    }

    public function testUpdateIncompleteFieldData()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'baz',
        ]);
        $content = GlobalContent::factory()->create();
        $response = $this->putJson(route('content.global-elements.update', $content), [
            'name' => 'foo',
            'element_id' => $element->id,
            'content' => 'something',
        ]);

        $response->assertStatus(422);
        $response = $this->putJson(route('content.global-elements.update', $content), [
            'name' => 'foo',
            'element_id' => $element->id,
            'content' => [
                [
                    'name' => 'foo',
                    'value' => 'something',
                ],
            ],
        ]);

        $response->assertStatus(422);
    }

    public function testCopyWithoutContent()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'baz',
        ]);
        $content = GlobalContent::factory()
            ->create([
                'element_id' => $element->id,
                'content' => ['foo' => 'something']
            ]);
        $response = $this->postJson(route('content.global-elements.copy', $content), [
            'name' => 'baz',
            'with_content' => false,
        ]);

        $response->assertStatus(201);
        $copy = GlobalContent::query()->findOrFail($response->json('data.id'));
        $this->assertEquals($element->id, $copy->element_id);
        $this->assertNull($copy->content);
    }

    public function testCopyWithContent()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);
        ContentElementField::factory()->create([
            'element_id' => $element->id,
            'name' => 'baz',
        ]);
        $content = GlobalContent::factory()
            ->create([
                'element_id' => $element->id,
                'content' => [
                    [
                        'name' => 'foo',
                        'value' => 'something',
                    ],
                ],
            ]);
        $response = $this->postJson(route('content.global-elements.copy', $content), [
            'name' => 'baz',
            'with_content' => true,
        ]);

        $response->assertStatus(201);
        $copy = GlobalContent::query()->findOrFail($response->json('data.id'));
        $this->assertEquals($element->id, $copy->element_id);
        $this->assertEquals([[
            'name' => 'foo',
            'value' => 'something',
        ]], $copy->content);
    }
}
